#
#     Copyright (C) 2016 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#
import os

from ccpem_core.ccpem_utils import ccpem_argparser
from ccpem_core.settings import which
from ccpem_core import process_manager
from ccpem_core.tasks import task_utils
from ccpem_core.tasks.refmac import refmac_task
from ccpem_core.tasks.nautilus import nautilus_results

class Buccaneer(task_utils.CCPEMTask):
    '''
    CCPEM Buccaneer task.
    '''
    task_info = task_utils.CCPEMTaskInfo(
        name='Buccaneer',
        author='Cowtan K',
        version='1.6.5',
        description=(
            'Buccaneer performs statistical chain tracing by identifying '
            'connected alpha-carbon positions using a likelihood-based density '
            'target.\n'
            'The target distributions are generated by a simulation '
            'calculation using a known reference structure for which '
            'calculated phases are available. The success of the method is '
            'dependent on the features of the reference structure matching '
            'those of the unsolved, work structure. For almost all cases, a '
            'single reference structure can be used, with modifications '
            'automatically applied to the reference structure to match its '
            'features to the work structure.  N.B. requires CCP4.'),
        short_description=(
            'Automated model building.  Requires CCP4'),
        documentation_link='http://www.ccp4.ac.uk/html/cbuccaneer.html',
        references=None)

    commands = {
        'cbuccaneer': which('cbuccaneer'),
        'refmac': which('refmac5'),
        'sftools': which('sftools'),
        'cad': which('cad'),
        'freerflag': which('freerflag')
        }

    # No longer need to set atomsf since refmac job defaults to use
    # "source EM MB" instead.
#    atomsf = os.path.join(
#        os.environ['CLIBD'],
#        'atomsf_electron.lib')


    def __init__(self,
                 database_path=None,
                 args=None,
                 args_json=None,
                 pipeline=None,
                 job_location=None,
                 verbose=False,
                 parent=None):
        super(Buccaneer, self).__init__(
            database_path=database_path,
            args=args,
            args_json=args_json,
            pipeline=pipeline,
            job_location=job_location,
            parent=parent)

    def parser(self):
        parser = ccpem_argparser.ccpemArgParser()
        #
        parser.add_argument(
            '-job_title',
            '--job_title',
            help='Short description of job',
            metavar='Job title',
            type=str,
            default=None)
        #
        parser.add_argument(
            '-mapin',
            '--input_map',
            help='''Target input map (mrc format)''',
            type=str,
            metavar='Input map',
            default=None)
        #
        parser.add_argument(
            '-resolution',
            '--resolution',
            help='''Resolution of input map (Angstrom)''',
            metavar='Resolution',
            type=float,
            default=None)
        #
        parser.add_argument(
            '-input_seq',
            '--input_seq',
            help='Input sequence file in any common format (e.g. pir, fasta)',
            type=str,
            default=None)
        #
        parser.add_argument(
            '-extend_pdb',
            '--extend_pdb',
            help='Initial PDB model to extend (pdb format)',
            type=str,
            default=None)
        #
        parser.add_argument(
            '-ncycle',
            '--ncycle',
            help='Number of Buccaneer pipeline cycles',
            metavar='Build cycles',
            type=int,
            default=5)
        #
        parser.add_argument(
            '-ncycle_refmac',
            '--ncycle_refmac',
            help='Number of refmac cycles',
            metavar='Refine cycles',
            type=int,
            default=10)
        #
        parser.add_argument(
            '-ncycle_buc1st',
            '--ncycle_buc1st',
            help='Number of Buccaneer cycles in 1st pipeline cycle',
            metavar='1st Buccaneer cycles',
            type=int,
            default=5)
        #
        parser.add_argument(
            '-ncycle_bucnth',
            '--ncycle_bucnth',
            help='Number of Buccaneer cycles in subsequent pipeline cycle',
            metavar='N-th Buccaneer cycles',
            type=int,
            default=3)
        #
        parser.add_argument(
            '-map_sharpen',
            '--map_sharpen',
            metavar='Map sharpen',
            help=('B-factor to apply to map. Negative B-factor to sharpen, '
                  'positive to blur, zero to leave map as input.'),
            type=float,
            default=0)
        #
        parser.add_argument(
            '-keywords',
            '--keywords',
            help='Keywords for advanced options.  Select file or define text',
            type=str,
            metavar='Keywords',
            default='')
        #
        parser.add_argument(
            '-refmac_keywords',
            '--refmac_keywords',
            help='Refmac keywords for advanced options. Select file or define text',
            type=str,
            metavar='Refmac Keywords',
            default='')
        return parser

    def run_pipeline(self, job_id=None, run=True, db_inject=None):
        '''
        Generate job classes and process.  Run=false for reloading.
        '''
        # Convert map to mtz (refmac)
        bfactor = self.args.map_sharpen()
        if bfactor >= 0:
            sharp_array = None
            blur_array = [bfactor]
        else:
            sharp_array = [-1 * bfactor]
            blur_array = None
        self.process_maptomtz = refmac_task.RefmacMapToMtz(
            command=self.commands['refmac'],
            resolution=self.args.resolution.value,
            mode='Global',
            name='Map to MTZ',
            job_location=self.job_location,
            map_path=self.args.input_map.value,
            blur_array=blur_array,
            sharp_array=sharp_array)
            #atomsf_path=self.atomsf)
        pl = [[self.process_maptomtz.process]]

        # Set MTZ SigF and FOM with SFTools
        self.process_sftools_buccanneer = SFToolsBuccaneer(
            command=self.commands['sftools'],
            job_location=self.job_location,
            hklin=self.process_maptomtz.hklout_path,
            name='Set MTZ SigF and SG')
        pl.append([self.process_sftools_buccanneer.process])

        # Set column labels with Cad
        self.process_cad_labels = CADBuccaneerLabels(
            job_location=self.job_location,
            resolution=self.args.resolution.value,
            hklin=self.process_sftools_buccanneer.hklout,
            sharpen_bfactor=self.args.map_sharpen.value,
            command=self.commands['cad'],
            name='Set MTZ labels')
        pl.append([self.process_cad_labels.process])

        # Create R free flags
        hklout = os.path.join(self.job_location,
                              'buccaneer.mtz')
        self.process_free_r_flags = FreeRFlags(
            command=self.commands['freerflag'],
            job_location=self.job_location,
            hklin=self.process_cad_labels.hklout,
            hklout=hklout,
            name='Set Rfree')
        pl.append([self.process_free_r_flags.process])

        # Save seq if sequence string is provided rather than file path
        if (not os.path.exists(self.args.input_seq.value) and
                isinstance(self.args.input_seq.value, str)):
            path = os.path.join(self.job_location,
                                'input.seq')
            f = open(path, 'w')
            f.write(self.args.input_seq.value)
            f.close()
            self.args.input_seq.value = f.name

        # Run Buccaneer - new pipeline = Buccaneer->Refmac refine pipeline->repeat...
        # Add optional refmac keywords
        refine_keywords = add_refmac_keywords(
            self.args.refmac_keywords.value, 
            self.process_cad_labels.label_out_suffix)

        for i in range(1, (self.args.ncycle.value + 1)) :
            # 1st cycle Buccaneer
            if i == 1 :
                self.process_buccaneer_pipeline = BuccaneerPipeline(
                    command=self.commands['cbuccaneer'],
                    job_location=self.job_location,
                    hklin=self.process_free_r_flags.hklout,
                    seqin=self.args.input_seq.value,
                    label_out_suffix=self.process_cad_labels.label_out_suffix,
                    ncycle=i,
                    buc_cycle=self.args.ncycle_buc1st.value,
                    resolution=self.args.resolution.value,
                    pdbin=self.args.extend_pdb.value,
                    pdbout=None,
                    name='Buccaneer build ' + str(i),
                    job_title=self.args.job_title.value,
                    keywords=self.args.keywords.value)
                pl.append([self.process_buccaneer_pipeline.process])

                # Run RefmacRefine (global)
                self.process_refine = refmac_task.RefmacRefine(
                    command=self.commands['refmac'],
                    job_location=self.job_location,
                    pdb_path=self.process_buccaneer_pipeline.pdbout,
                    mtz_path=self.process_free_r_flags.hklout,
                    pdbout_path=os.path.join(self.job_location, 'refined' + str(i) + '.pdb'),
                    resolution=self.args.resolution.value,
                    mode='Global',
                    name='Refmac refine (global) ' + str(i),
                    sharp=self.args.map_sharpen.value,
                    ncycle=self.args.ncycle_refmac.value,
                    output_hkl=True,
                    keywords=refine_keywords)
                pl.append([self.process_refine.process])
            else :
                #nth cycle Buccaneer
                self.process_buccaneer_pipeline = BuccaneerPipeline(
                            command=self.commands['cbuccaneer'],
                            job_location=self.job_location,
                            hklin=self.process_refine.hklout_path,
                            seqin=self.args.input_seq.value,
                            label_out_suffix=self.process_cad_labels.label_out_suffix,
                            ncycle=i,
                            buc_cycle=self.args.ncycle_bucnth.value,
                            resolution=self.args.resolution.value,
                            pdbin=self.process_refine.pdbout_path,
                            pdbout=None,
                            name='Buccaneer build ' + str(i),
                            job_title=self.args.job_title.value,
                            keywords=self.args.keywords.value)
                pl.append([self.process_buccaneer_pipeline.process])

                # Run RefmacRefine (global)
                self.process_refine = refmac_task.RefmacRefine(
                    command=self.commands['refmac'],
                    job_location=self.job_location,
                    pdb_path=self.process_buccaneer_pipeline.pdbout,
                    mtz_path=self.process_free_r_flags.hklout,
                    pdbout_path=os.path.join(self.job_location, 'refined' + str(i) + '.pdb'),
                    resolution=self.args.resolution.value,
                    mode='Global',
                    name='Refmac refine (global) ' + str(i),
                    sharp=self.args.map_sharpen.value,
                    ncycle=self.args.ncycle_refmac.value,
                    output_hkl=True,
                    keywords=refine_keywords)
                pl.append([self.process_refine.process])

        custom_finish = BuccaneerResultsOnFinish(
            pipeline_path=self.job_location+'/task.ccpem',
            refine_process=self.process_refine)

        if run:
            os.chdir(self.job_location)
            self.pipeline = process_manager.CCPEMPipeline(
                pipeline=pl,
                job_id=job_id,
                args_path=self.args.jsonfile,
                location=self.job_location,
                database_path=self.database_path,
                db_inject=db_inject,
                taskname=self.task_info.name,
                title=self.args.job_title.value,
                verbose=self.verbose,
                on_finish_custom=custom_finish)
            self.pipeline.start()


    def validate_args(self):
        # Now do at gui level
        return True

class SFToolsBuccaneer(object):
    '''
    STFTools process to set space group to p1 and SIGFem SIGFem columns
    '''
    def __init__(self,
                 job_location,
                 hklin,
                 command=which('sftools'),
                 name=None):
        assert command is not None
        self.job_location = job_location
        self.hklin = hklin
        self.hklout = os.path.join(job_location,
                                   'sftools.mtz')
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__
        self.stdin = None
        #
        self.set_args()
        self.set_stdin()
        #
        self.process = process_manager.CCPEMProcess(
            name=self.name,
            command=command,
            args=self.args,
            location=self.job_location,
            stdin=self.stdin)

    def set_args(self):
        self.args = []

    def set_stdin(self):
        self.stdin = '''
READ {0}
SET SPACEGROUP
P 1
CALC Q COL SIGFem = 1
CALC W COL FOMem = 0.5
WRITE {1}
END'''.format(self.hklin, self.hklout)


class CADBuccaneerLabels(object):
    '''
    CAD process to change Fout<0> etc column labels.
        N.B. Refmac produces Fout<0> labels but buccaneer can't interpret these
        label names.
    '''
    def __init__(self,
                 job_location,
                 hklin,
                 resolution,
                 command=which('cad'),
                 sharpen_bfactor=0.0,
                 name=None):
        assert command is not None
        self.job_location = job_location
        self.hklin = hklin
        self.resolution = resolution
        if sharpen_bfactor is None:
            sharpen_bfactor = 0
        self.sharpen_bfactor = float(sharpen_bfactor)
        self.hklout = os.path.join(job_location,
                                   'cad_labels.mtz')
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__
        self.stdin = None
        #
        self.set_args()
        self.set_stdin()
        #
        self.process = process_manager.CCPEMProcess(
            name=self.name,
            command=command,
            args=self.args,
            location=self.job_location,
            stdin=self.stdin)

    def set_args(self):
        self.args  = ['hklin1', self.hklin]
        self.args += ['hklout', self.hklout]

    def set_stdin(self):
#         if self.sharpen_bfactor == 0 or self.sharpen_bfactor is None:
#             self.stdin = '''
# labin file 1 -
#     E1 = Fout0 -
#     E2 = SIGFem -
#     E3 = Pout0 -
#     E4 = FOMem
# labout file 1 -
#     E1 = Fem -
#     E2 = SIGFem -
#     E3 = PHIem -
#     E4 = FOMem
# ctypin file 1 -
#     E1 = F -
#     E2 = Q -
#     E3 = P -
#     E4 = W
# resolution file 1 999 {0}'''.format(self.resolution)
#             self.label_out_suffix = 'Fem'
#             return
        if self.sharpen_bfactor == 0:
            label_in_suffix = str(0)
            self.label_out_suffix = 'SHARP{0:.0f}'.format(self.sharpen_bfactor)
        elif self.sharpen_bfactor >= 0:
            label_in_suffix = 'Blur_{0:.2f}'.format(self.sharpen_bfactor)
            self.label_out_suffix = 'BLUR{0:.0f}'.format(self.sharpen_bfactor)
        else:
            self.sharpen_bfactor *=-1
            label_in_suffix = 'Sharp_{0:.2f}'.format(self.sharpen_bfactor)
            self.label_out_suffix = 'SHARP{0:.0f}'.format(self.sharpen_bfactor)
        self.stdin = '''
labin file 1 -
    E1 = Fout0 -
    E2 = SIGFem -
    E3 = Fout{0} -
    E4 = SIGFem -
    E5 = Pout0 -
    E6 = FOMem
labout file 1 -
    E1 = Fem -
    E2 = SIGFem -
    E3 = Fem{1} -
    E4 = SIGFem{1} -
    E5 = PHIem -
    E6 = FOMem
ctypin file 1 -
    E1 = F -
    E2 = Q -
    E3 = F -
    E4 = Q -
    E5 = P -
    E6 = W
resolution file 1 999 {2}'''.format(label_in_suffix,
                                    self.label_out_suffix,
                                    self.resolution)


class FreeRFlags(object):
    '''
    Tags each reflection in an MTZ file with a flag for cross-validation.
    '''
    def __init__(self,
                 job_location,
                 hklin,
                 command = which('freerflag'),
                 hklout=None,
                 free_r_frac=0.05,
                 name=None):
        assert command is not None
        self.job_location = job_location
        self.hklin = hklin
        self.hklout = hklout
        if self.hklout is None:
            self.hklout = os.path.join(job_location,
                                        'freerflags.mtz')
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__
        self.free_r_frac = free_r_frac
        self.stdin = None
        #
        self.set_args()
        self.set_stdin()
        #
        self.process = process_manager.CCPEMProcess(
            name=self.name,
            command=command,
            args=self.args,
            location=self.job_location,
            stdin=self.stdin)

    def set_args(self):
        self.args  = ['hklin', self.hklin]
        self.args += ['hklout', self.hklout]

    def set_stdin(self):
        self.stdin = '''
FREERFRAC {0}
'''.format(self.free_r_frac)


class BuccaneerPipeline(object):
    '''
    Statistical protein chain tracing (N.B. runs cbuccaneer)
    '''
    def __init__(self,
            command,
            job_location,
            hklin,
            seqin,
            label_out_suffix,
            ncycle=1,
            buc_cycle=5,
            resolution=2.0,
            pdbin=None,
            pdbout=None,
            name=None,
            job_title=None,
            keywords=None):
        # counter for overall BuccaneerPipeline cycles, start with 1
        assert command is not None
        self.job_location = job_location
        self.hklin = hklin
        self.seqin = seqin
        assert os.path.exists(path=self.seqin)
        self.label_out_suffix = label_out_suffix
        self.pdbout = pdbout
        self.ncycle = ncycle
        self.buc_cycle = buc_cycle
        self.resolution = resolution
        self.pdbin = pdbin
        if self.pdbout is None:
            self.pdbout = os.path.join(job_location, 'build' + str(ncycle) + '.pdb')
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__
        self.job_title = job_title
        self.stdin = None
        self.stdin_extra = None
        self.keywords = keywords
        #
        self.set_args()
        self.set_stdin()
        #
        self.process = process_manager.CCPEMProcess(
                name=self.name,
                command=command,
                args=self.args,
                location=self.job_location,
                stdin=self.stdin)

    def set_args(self):
        self.args = ['-stdin']

    def set_stdin(self):
        pdbin_ref = os.path.join(os.environ['CLIBD'],
            'reference_structures/reference-1tqw.pdb')
        mtzin_ref = os.path.join(os.environ['CLIBD'],
            'reference_structures/reference-1tqw.mtz')
        if self.ncycle == 1:    # 1st buccaneer pipeline cycle
            self.stdin_extra = '''colin-phifom PHIem, FOMem
            '''
        else:                   # nth buccaneer pipeline cycle
            self.stdin_extra = '''colin-hl HLACOMB, HLBCOMB, HLCCOMB, HLDCOMB
colin-fc FWT, PHWT
'''

        self.stdin = '''
title {0}
pdbin-ref {1}
mtzin-ref {2}
colin-ref-fo FP.F_sigF.F,FP.F_sigF.sigF
colin-ref-hl FC.ABCD.A,FC.ABCD.B,FC.ABCD.C,FC.ABCD.D
seqin {3}
mtzin {4}
colin-fo Fem{5},SIGFem{5}
colin-free FreeR_flag
pdbout {6}
cycles {7}
anisotropy-correction
fast
correlation-mode
sequence-reliability 0.95
new-residue-name UNK
resolution {8}
model-filter-sigma 3.0
mr-model-filter-sigma 3.0
xmlout program{9}.xml
'''.format(self.job_title,
       pdbin_ref,
       mtzin_ref,
       self.seqin,
       self.hklin,
       self.label_out_suffix,
       self.pdbout,
       self.buc_cycle,
       self.resolution,
       self.ncycle)
        #
        self.stdin = self.stdin + self.stdin_extra
        # Add optional buccaneer keywords
        if isinstance(self.keywords, str):
            if self.keywords != '':
                # Remove trailing white space and new lines
                keywords = self.keywords.strip()
                for line in keywords.split('\n'):
                    self.stdin += line + '\n'
        if self.pdbin is not None:
            self.stdin += 'pdbin {0}\n'.format(self.pdbin)

class BuccaneerResultsOnFinish(process_manager.CCPEMPipelineCustomFinish):
    '''
    Generate RVAPI results on finish.
    '''
    def __init__(self,
                 pipeline_path,
                 refine_process):
        super(BuccaneerResultsOnFinish, self).__init__()
        self.pipeline_path = pipeline_path
        self.refine_process = refine_process

    def on_finish(self, parent_pipeline=None):
        #generate RVAPI report
        nautilus_results.PipelineResultsViewer(
                  pipeline_path=self.pipeline_path)

def add_refmac_keywords(refkeywords, cad_labelout_suffix):
    '''
    Add default and additional refmac keywords
    '''
    refine_keywords='''labin PHIB=PHIem FOM=FOMem FREE=FreeR_flag FP=Fem{0} SIGFP=SIGFem{0}
PHOUT
PNAME buccaneer
DNAME buccaneer
'''.format(cad_labelout_suffix)
    
    if isinstance(refkeywords, str):
        if refkeywords != '':
            # Remove trailing white space and new lines
            keywords = refkeywords.strip()
            for line in keywords.split('\n'):
                refine_keywords += line + '\n'

    return refine_keywords

def main():
    '''
    Run task
    '''
    task_utils.command_line_task_launch(
        task=Buccaneer)

if __name__ == '__main__':
    main()
