#
#     Copyright (C) 2018 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#


# XXX TODO
# 
# Get basic example working using np_locscale branch
# Possible use multiprocessing as alternative to mpi4py?
# Add setup.py for ease of individual distribution and install via ccp-em

import os
from loc_scale import np_locscale
from ccpem_core.ccpem_utils import ccpem_argparser
from ccpem_core import ccpem_utils
from ccpem_core import process_manager
from ccpem_core.tasks import task_utils
from ccpem_core.tasks.refmac import refmac_task


class LocScale(task_utils.CCPEMTask):
    '''
    CCPEM LocScale Task.
    '''
    task_info = task_utils.CCPEMTaskInfo(
        name='LocScale',
        author='A. J. Jakobi, M. Wilmanns, C. Sachse',
        version='1.1',
        description=(
            'Local amplitude sharpening based target structure'),
        short_description=(
            '[BETA TEST] Local amplitude sharpening based target structure'
            'Coefficient'),
        documentation_link='https://elifesciences.org/articles/27131',
        references=None)
    commands = {
        'flexem_python': ['ccpem-python',
                          os.path.realpath(np_locscale.__file__)]}


    def __init__(self,
                 database_path=None,
                 args=None,
                 args_json=None,
                 pipeline=None,
                 job_location=None,
                 parent=None):
        super(LocScale, self).__init__(database_path=database_path,
                                       args=args,
                                       args_json=args_json,
                                       pipeline=pipeline,
                                       job_location=job_location,
                                       parent=parent)

    def parser(self):
        parser = ccpem_argparser.ccpemArgParser()
        #
        job_title = parser.add_argument_group()
        job_title.add_argument(
            '-job_title',
            '--job_title',
            help='Short description of job',
            metavar='Job title',
            type=str,
            default=None)
        #
        job_location = parser.add_argument_group()
        job_location.add_argument(
            '-job_location',
            '--job_location',
            help='Directory to run job',
            metavar='Job location',
            type=str,
            default=None)
        #
        target_map = parser.add_argument_group()
        target_map.add_argument(
            '-target_map',
            '--target_map',
            help='Target map to be auto sharpened (mrc format)',
            metavar='Target map',
            type=str,
            default=None)
        #
        reference_model_map = parser.add_argument_group()
        reference_model_map.add_argument(
            '-reference_model_map',
            '--reference_model_map',
            help='Reference map from model (mrc format)',
            metavar='Reference map',
            type=str,
            default=None)
        #
        mask_map = parser.add_argument_group()
        mask_map.add_argument(
            '-mask_map',
            '--mask_map',
            help='Mask map (mrc format)',
            metavar='Mask map',
            type=str,
            default=None)
        #
        parser.add_argument(
            '-resolution',
            '--resolution',
            help='Resolution of target map (Angstrom)',
            metavar='Resolution',
            type=float,
            default=None)
        #
        parser.add_argument(
            '-pixel_size',
            '--pixel_size',
            help='Pixel size in Angstrom',
            metavar='Pixel size',
            type=float,
            default=None)
        #
        parser.add_argument(
            '-reference_model',
            '--reference_model',
            help='Reference model (PDB format)',
            metavar='Reference model',
            type=str,
            default=None)
        # 
        parser.add_argument(
            '-refine_bfactors',
            '--refine_bfactors',
            help='Refine reference structure B-factors using Refmac',
            metavar='Refine B-factors',
            type=bool,
            default=True)
        #
        parser.add_argument(
            '-window_size',
            '--window_size',
            help=('Window size in pixel, if not given 7 * map resolution '
                  '/ pixel size used'),
            metavar='Window size',
            type=int,
            default=None)
        #
        parser.add_argument(
            '-use_mpi',
            '--use_mpi',
            help='Use mpi for parallel processing',
            metavar='Use MPI',
            type=bool,
            default=False)
        #
        parser.add_argument(
            '-n_mpi',
            '--n_mpi',
            help='Number of mpi nodes',
            metavar='MPI nodes',
            type=float,
            default=1)
        #
        return parser

    def run_pipeline(self, job_id=None, db_inject=None):
#         # Run Refmac to refine reference structure B-factors
#         if self.args.refine_bfactors():
#             self.refmac_process = refmac_task.RefmacRefine(
#                 command=self.command,
#                 job_location=self.job_location,
#                 pdb_path=self.args.reference_model(),
#                 mtz_path=self.process_free_r_flags.hklout,
#                 resolution=self.args.resolution.value,
#                 mode='Global',
#                 name='Refmac refine (global)',
#                 sharp=None,
#                 ncycle=20,
#                 output_hkl=True)
#             pl.append([self.refmac_process.process])
#             # XXX TODO - get refmac structure
#             loc_scale_structure = ''
#         else:
#             loc_scale_structure = self.args.reference_model

        # Calculate Fcalc
        self.refmac_sfcalc_crd_process = refmac_task.RefmacSfcalcCrd(
            job_location=self.job_location,
            pdb_path=self.args.reference_model(),
            resolution=self.args.resolution())
        self.refmac_sfcalc_mtz = os.path.join(
            self.job_location,
            'sfcalc_from_crd.mtz')

        # Generate process
        self.loc_scale_process = LocScaleWrapper(
            job_location=self.job_location,
            target_map=self.args.target_map(),
            resolution=self.args.resolution(),
            pixel_size=self.args.pixel_size(),
            reference_model_map=self.args.reference_model_map(),
            mask_map = self.args.mask_map(),
            output_map=None,
            window_size=self.args.window_size(),
            name='Loc Scale')

        pl = [[self.loc_scale_process.process]]
        # pipeline
        self.pipeline = process_manager.CCPEMPipeline(
            pipeline=pl,
            job_id=job_id,
            args_path=self.args.jsonfile,
            location=self.job_location,
            db_inject=db_inject,
            database_path=self.database_path,
            taskname=self.task_info.name,
            title=self.args.job_title.value)
        self.pipeline.start()

class LocScaleWrapper(object):
    '''
    Wrapper for LocScale process.
    '''
    commands = {
    'loc_scale_python': ['ccpem-python',
                         os.path.realpath(np_locscale.__file__)]}

    def __init__(self,
                 job_location,
                 target_map,
                 resolution,
                 pixel_size,
                 reference_model_map,
                 output_map=None,
                 mask_map=None,
                 window_size=None,
                 name=None):
        self.job_location = ccpem_utils.get_path_abs(job_location)
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__
        if window_size is None:
            window_size = int((7.0 * resolution) / pixel_size)
        if output_map is None:
            output_map = 'loc_scale.mrc'
        output_map = os.path.join(self.job_location,
                                  output_map)

        self.args = [
            '--em_map', os.path.abspath(target_map),
            '--model_map', os.path.abspath(reference_model_map),
            '--apix', pixel_size,
            '--window_size', window_size,
            '--outfile', output_map]

        if mask_map is not None:
            self.args += ['--mask', mask_map]

        # Set process
        self.process = process_manager.CCPEMProcess(
            name=self.name,
            command=self.commands['loc_scale_python'],
            args=self.args,
            location=self.job_location,
            stdin=None)

def main():
    '''
    Run task
    '''
    task_utils.command_line_task_launch(
        task=LocScale)

if __name__ == '__main__':
    main()
