#
#     Copyright (C) 2017 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#


'''
Task window for LocScale.
'''
import os
from ccpem_gui.utils import window_utils
from ccpem_core.ccpem_utils import ccpem_file_types
from ccpem_core.tasks.loc_scale import loc_scale_task

# XXX To Do
# Set modes:
# -> (1) Provide reference map
#    -> Need - reference map, target map, mask?
# -> (2) Provide reference PDB
#    -> Need - reference pdb, target map, mask?

class LocScaleWindow(window_utils.CCPEMTaskWindow):
    '''
    LocScale window.
    '''
    def __init__(self,
                 task,
                 parent=None):
        super(LocScaleWindow, self).__init__(task=task,
                                              parent=parent)

    def set_args(self):
        '''
        Set input arguments
        '''
        # Job title
        self.title_input = window_utils.TitleArgInput(
            parent=self,
            arg_name='job_title',
            args=self.args)
        self.args_widget.args_layout.addWidget(self.title_input)
        self.title_input.value_line.editingFinished.connect(
            self.handle_title_set)

        # Input target map
        target_map_input = window_utils.FileArgInput(
            parent=self,
            arg_name='target_map',
            args=self.args,
            file_types=ccpem_file_types.mrc_ext,
            required=True)
        self.args_widget.args_layout.addWidget(target_map_input)

        # Input target map
        reference_model_map_input = window_utils.FileArgInput(
            parent=self,
            arg_name='reference_model_map',
            args=self.args,
            file_types=ccpem_file_types.mrc_ext,
            required=True)
        self.args_widget.args_layout.addWidget(reference_model_map_input)

        # Input mask map
        mask_map_input = window_utils.FileArgInput(
            parent=self,
            arg_name='mask_map',
            args=self.args,
            file_types=ccpem_file_types.mrc_ext,
            required=True)
        self.args_widget.args_layout.addWidget(mask_map_input)

        # Resolution
        resolution_input = window_utils.NumberArgInput(
            parent=self,
            decimals=1,
            step=0.1,
            arg_name='resolution',
            args=self.args,
            required=True)
        self.args_widget.args_layout.addWidget(resolution_input)

        # Resolution
        pixel_input = window_utils.NumberArgInput(
            parent=self,
            decimals=1,
            step=0.1,
            arg_name='pixel_size',
            args=self.args,
            required=True)
        self.args_widget.args_layout.addWidget(pixel_input)

        # Reference model
        reference_map_input = window_utils.FileArgInput(
            parent=self,
            arg_name='reference_model',
            args=self.args,
            file_types=ccpem_file_types.mrc_ext,
            required=True)
        self.args_widget.args_layout.addWidget(reference_map_input)
#         refine_bfactor_input = window_utils.ChoiceArgInput(
#             parent=self,
#             arg_name='refine_bfactors',
#             args=self.args,
#             required=False)
#         self.args_widget.args_layout.addWidget(refine_bfactor_input)

        # Extended options
        extended_options_frame = window_utils.CCPEMExtensionFrame(
            button_name='Extended options',
            button_tooltip='Show extended options')
        self.args_widget.args_layout.addLayout(extended_options_frame)

        # Use MPI input
        use_mpi_input = window_utils.ChoiceArgInput(
            parent=self,
            arg_name='use_mpi',
            required=False,
            args=self.args)
        extended_options_frame.add_extension_widget(use_mpi_input)
        use_mpi_input.value_line.currentIndexChanged.connect(
            self.set_n_mpi_visible)

        self.n_mpi_input = window_utils.NumberArgInput(
            parent=self,
            arg_name='n_mpi',
            required=False,
            args=self.args)
        extended_options_frame.add_extension_widget(self.n_mpi_input)
        self.set_n_mpi_visible()

        # Set window size
        scale_input = window_utils.NumberArgInput(
            parent=self,
            arg_name='window_size',
            required=False,
            args=self.args)
        extended_options_frame.add_extension_widget(scale_input)

        # Set inputs for launcher
        self.launcher.add_file(
            arg_name='target_map',
            file_type='map',
            description=self.args.target_map.help,
            selected=True)
        self.launcher.add_file(
            arg_name='reference_model',
            file_type='pdb',
            description=self.args.reference_model.help,
            selected=True)

    def set_n_mpi_visible(self):
        if self.args.use_mpi():
            self.n_mpi_input.show()
        else:
            self.n_mpi_input.hide()

    def set_on_job_finish_custom(self):
        '''
        Actions to run on job completion.  Show LocScale map
        '''
        # Set launcher files
        output_map_path = os.path.join(self.task.job_location,
                                       'loc_scale.mrc')
        self.launcher.add_file(
            arg_name=None,
            path=output_map_path,
            file_type='map',
            description='LocScale scaled map',
            selected=True)
        self.launcher.set_tree_view()
        self.launcher_dock.raise_()

def main():
    '''
    Launch stand alone window.
    '''
    # Set args.. for testing
    from ccpem_core.test_data.tasks import loc_scale as loc_scale_td
    args_json = os.path.join(os.path.dirname(loc_scale_td.__file__),
                             'unittest_args.json')
    window_utils.standalone_window_launch(
        args_json=args_json,
        task=loc_scale_task.LocScale,
        window=LocScaleWindow)

if __name__ == '__main__':
    main()
