#
#     Copyright (C) 2015 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#

import unittest
import os
import sys
import shutil
import time
import tempfile
from PyQt4 import QtGui, QtCore
from PyQt4.QtTest import QTest
from ccpem_core.tasks.molrep import molrep_task
from ccpem_gui.tasks.molrep import molrep_window
from ccpem_core.test_data.tasks import molrep as molrep_test
from ccpem_core import ccpem_utils
from ccpem_core import process_manager

app = QtGui.QApplication(sys.argv)

class Test(unittest.TestCase):
    '''
    Unit test for Molrep (invokes GUI).
    '''
    def setUp(self):
        '''
        Setup test data and output directories.
        '''
        self.test_data = os.path.dirname(molrep_test.__file__)
        self.test_output = tempfile.mkdtemp()

    def tearDown(self):
        '''
        Remove temporary data.
        '''
        if os.path.exists(path=self.test_output):
            shutil.rmtree(self.test_output)

    def test_molrep_window_integration(self):
        '''
        Test Molrep pipeline via GUI.
        '''
        ccpem_utils.print_header(message='Unit test - Molrep')
        # Load args
        args_path = os.path.join(self.test_data, 'unittest_args.json')
        run_task = molrep_task.MolRep(job_location=self.test_output,
                                      args_json=args_path)
        self.run_molrep_task_in_window(run_task)
        
        # Check correlation factor can be found and is correct
        stdout = run_task.pipeline.pipeline[-1][0].stdout
        tail = ccpem_utils.tail(stdout, maxlen=30)
        posn = tail.find('Final CC')
        assert posn != -1
        cf = float(tail[posn:posn+30].split()[3])
        print 'Final CC: {}'.format(cf)
        self.assertGreater(cf, 0.96)

    def test_molrep_haemoglobin_with_target_sequence(self):
        '''
        Test Molrep with 5me2 haemoglobin map, 1mbn myoglobin model.
        
        This takes about 3 minutes to run, probably a bit long for a test...
        '''
        ccpem_utils.print_header(message='Unit test - Molrep with target sequence')
        # Load args
        args_path = os.path.join(self.test_data, 'target_seq_test_args.json')
        run_task = molrep_task.MolRep(job_location=self.test_output,
                                      args_json=args_path)
        self.run_molrep_task_in_window(run_task)
        
        # Check correlation factor can be found and is correct
        stdout = run_task.pipeline.pipeline[-1][0].stdout
        tail = ccpem_utils.tail(stdout, maxlen=10)
        posn = tail.find('Score')
        assert posn != -1
        cf = float(tail[posn:].split('\n')[1]
                              .split()[-1])
        print 'Score: {}'.format(cf)
        self.assertGreater(cf, 0.39)

    def test_molrep_haemoglobin_with_partial_fixed_model(self):
        '''
        Test Molrep with 5me2 map and fixed AB chain model, 1mbn search model.
        
        This takes about 3XX minutes to run, probably a bit long for a test...
        '''
        ccpem_utils.print_header(message='Unit test - Molrep with fixed model')
        # Load args
        args_path = os.path.join(self.test_data, 'fixed_model_test_args.json')
        run_task = molrep_task.MolRep(job_location=self.test_output,
                                      args_json=args_path)
        self.run_molrep_task_in_window(run_task)
        
        # Check correlation factor can be found and is correct
        stdout = run_task.pipeline.pipeline[-1][0].stdout
        tail_1 = ccpem_utils.tail(stdout, maxlen=60)
        posn_1 = tail_1.find('Score')
        assert posn_1 != -1
        score_1 = float(tail_1[posn_1:].split('\n')[1]
                                     .split()[-1])
        print 'Score (first solution): {}'.format(score_1)
        self.assertGreater(score_1, 0.39)
        
        tail_2 = ccpem_utils.tail(stdout, maxlen=10)
        posn_2 = tail_2.find('Score')
        assert posn_2 != -1
        score_2 = float(tail_2[posn_2:].split('\n')[1]
                                       .split()[-1])
        print 'Score (second solution): {}'.format(score_2)
        self.assertGreater(score_2, 0.35)

    def run_molrep_task_in_window(self, run_task, timeout_limit=500):
        '''
        Run a Molrep task via the GUI.
        '''
        # Setup GUI
        self.window = molrep_window.MolrepWindow(task=run_task)
        # Mouse click run
        QTest.mouseClick(
            self.window.tool_bar.widgetForAction(self.window.tb_run_button),
            QtCore.Qt.LeftButton)
        # Wait for run to complete
        self.job_completed = False
        timeout = 0
        stdout = run_task.pipeline.pipeline[-1][0].stdout
        # stdout = sfcheckmapvsmodel_stdout.txt
        while not self.job_completed and timeout < timeout_limit:
            print 'Molrep running for {0} secs (timeout = {1})'.format(timeout, timeout_limit)
            time.sleep(5.0)
            timeout += 5
            status =\
                process_manager.get_process_status(run_task.pipeline.json)
            if status == 'finished':
                if os.path.isfile(stdout):
                    tail = ccpem_utils.tail(stdout, maxlen=10)
                    if tail.find('CCP-EM process finished') != -1:
                        self.job_completed = True
            elif status == 'failed':
                print 'Molrep job failed'
                break
        # Check timeout
        assert timeout < timeout_limit
        # Check job completed
        assert self.job_completed
        # Check output pdb created
        assert os.path.exists(os.path.join(self.test_output,
                                           'molrep.pdb'))

if __name__ == '__main__':
    unittest.main()
