
import pyrvapi_ext as API
import regex_tree as RT
import time
import sys


class molrep_parser(object):

    def flush(self):
        if self.pause > 0:
            time.sleep(self.pause)

        API.flush()

    def __init__(
        self,
        rvapi_grid,
        show_progs = False,
        pause = 0,
        **kwargs
    ):
        self.pause = pause

        i = ' +(-?[0-9]+)'
        r = ' +(-?[0-9]+\.[0-9]*)'
        x = ' +\((-?[0-9]+\.[0-9]*)\)'
        b = '^'
        v = ' *\|'
        e = ' *\n'
        crf_title = '^     --- Peaks of Rotation Function ---\n'
        tf_title = '^  --- Translation function ---\n'
        crf_head = '^ +\| +theta +phi +chi +Rf/sigma \|\n'
        tf_head = '^ +RF +TF +TF/sig +TFcntrst +PFind +PF'
        tf_head += ' +PF(?:min|mod) +wRfac +Score +Cntrst +\(for\)\n'
        coord_conv = '^ --- convert "molrep.crd" to "molrep.pdb" ---\n'

        crf_startline = RT.LogDataLine(crf_head)
        crf_dataline = RT.LogDataLine(b + v + i + 4* r + v + e)
        crf_finishline = RT.LogDataLine('^ \+-+\+\n')
        tf_startline = RT.LogDataLine(tf_head)
        tf_dataline = RT.LogDataLine(b + 2* i + 8* r + x + e)
        tf_finishline = RT.LogDataLine('^ \+-+\+\n')
        success = RT.LogDataLine(coord_conv)

        self.parser = RT.LogDataParser()
        self.parser.add_next(crf_startline)
        crf_startline.add_next(crf_dataline)
        crf_dataline.add_next(crf_dataline, crf_finishline)
        crf_finishline.add_next(tf_startline)
        tf_startline.add_next(tf_dataline)
        tf_dataline.add_next(tf_dataline, tf_finishline)
        tf_finishline.add_next(tf_startline, success)

        crf_startline.add_action(self.crf_started)
        crf_dataline.add_action(self.crf_data)
        crf_finishline.add_action(self.crf_finished)
        tf_startline.add_action(self.tf_started)
        tf_dataline.add_action(self.tf_data)
        tf_finishline.add_action(self.tf_finished)
        success.add_action(self.job_succeeded)

        self.show_progs = show_progs
        self.sect = rvapi_grid
        if show_progs:
            self.sect = API.pyrvapi_section(rvapi_grid, 'Molrep', 0, 0, 1, 1)

        w = API.loggraph(self.sect, 0, 0, 1, 1)

        f = API.graph_data(w, 'Rotation Function')
        self.plot_rf = MLP_plot_rf(f, w)

        f = API.graph_data(w, 'Translation Function')
        self.plot_tfc = MLP_plot_tfc(f, w)
        self.plot_tfz = MLP_plot_tfz(f, w)

        self.output = MLP_show_output()
        self.jobdict = dict()

    def parse_stream(self, istream, ostream=None, verbose=False, pause=0, patches=None):
        self.parser.parse_stream(istream, ostream, verbose, pause, patches)
        if self.show_progs:
            self.sect.set_state(False)

        self.flush()

    def crf_started(self, items):
        self.plot_rf.start(items, self.jobdict)

    def crf_data(self, items):
        self.plot_rf.add_data(items, self.jobdict)
        self.flush()

    def crf_finished(self, items):
        self.plot_rf.finish(items, self.jobdict)

    def tf_started(self, items):
        self.plot_tfc.start(items, self.jobdict)
        self.plot_tfz.start(items, self.jobdict)

    def tf_data(self, items):
        self.plot_tfc.add_data(items, self.jobdict)
        self.flush()
        self.plot_tfz.add_data(items, self.jobdict)
        self.flush()

    def tf_finished(self, items):
        self.plot_tfc.finish(items, self.jobdict)
        self.plot_tfz.finish(items, self.jobdict)

    def job_succeeded(self, items):
        self.output.info(items, self.jobdict)


class MLP_plot_rf(object):

    def __init__(self, f, w):
        self.x_index = 0
        self.y_index = 4

        p = API.graph_plot(w, pltTitle='RF based Z-score', xName='RF peak No', yName='RF/sigRF')
        p.set_xrange(0, 44)
        p.set_yrange(-0.02, 0.6)
        self.x = API.graph_dataset(f, 'RF peak No', 'Rotation Function Peak Rank')
        self.y = API.graph_dataset(f, 'RF/sigRF', 'RF based Z-score', False)
        API.plot_line(p, f, self.x, self.y)

    def start(self, items, jobdict):
        jobdict['np_rf'] = 0

    def add_data(self, items, jobdict):
        self.x.add_datum(int(items[self.x_index]))
        self.y.add_datum(float(items[self.y_index]))
        jobdict['np_rf'] += 1

    def finish(self, items, jobdict):
        pass


class MLP_plot_tf(object):

    def __init__(self, f, w):
        self.x_index = 0
        self.cou = 0
        self.f = f

        self.p = API.graph_plot(w, **self.p_kwargs)
        self.p.set_legend(API.pyrvapi.RVAPI_LEGEND_LOC_SE, API.pyrvapi.RVAPI_LEGEND_PLACE_Inside)

    def start(self, items, jobdict):
        self.cou += 1
        self.p.set_xrange(0, jobdict['np_rf'])

        y_kwargs = dict(
            setName = self.y_fmt['name'] %self.cou,
            setHeader = self.y_fmt['header'] %self.cou,
            isint=False
        )

        self.x = API.graph_dataset(self.f, **self.x_kwargs)
        self.y = API.graph_dataset(self.f, **y_kwargs)
        API.plot_line(self.p, self.f, self.x, self.y)

    def add_data(self, items, jobdict):
        self.x.add_datum(int(items[self.x_index]))
        self.y.add_datum(float(items[self.y_index]))

    def finish(self, items, jobdict):
        pass

    x_kwargs = dict(
        setName='RF peak No',
        setHeader='Rotation Function Peak Rank',
    )


class MLP_plot_tfc(MLP_plot_tf):

    y_index = 8

    p_kwargs = dict(
        pltTitle='Correlation Coefficient',
        xName='RF peak No',
        yName='CC',
    )

    y_fmt = dict(
        name = 'TF CC for copy %d',
        header = 'TF Correlation Coefficient for copy %d',
    )


class MLP_plot_tfz(MLP_plot_tf):

    y_index = 2

    p_kwargs = dict(
        pltTitle='TF based Z-score',
        xName='RF peak No',
        yName='TF/sigTF',
    )

    y_fmt = dict(
        name = 'TF/sigTF for copy %d',
        header = 'TF based Z-score for copy %d',
    )


class MLP_show_output(object):

    def info(self, items, jobdict):
        pass


