
import os, sys
import math
import time
import argparse
import subprocess as SP

import pyrvapi_ext as API

class logWriter(object):

  def __init__(self, logfile, nrecords):
    lst = [logfile]
    for i in range(1, 1 + nrecords):
      lst.append(' [%4i] Record number %d .....\n' %(i, i))

    self.content = ''.join(lst)
    self.lastPos = 0
    self.ostream = open(logfile, 'w')

  def dump_fraction(self, p):
   length = len(self.content)
   currentPos = self.lastPos
   self.lastPos = currentPos + int(length* p)
   if self.lastPos > length:
      self.lastPos = length

   if self.lastPos > currentPos:
      self.ostream.write(self.content[currentPos:self.lastPos])
      self.ostream.flush()

def start_jsrview(reportdir, ccp4, html='index.html', ostream=sys.stdout):
  jsrview_fname = 'jsrview.exe' if sys.platform.startswith('win') else 'jsrview'
  jsrview_exe = os.path.join(ccp4, 'libexec', jsrview_fname)
  jsrv_stdo = os.path.join(reportdir, 'stdout.txt')
  jsrv_stde = os.path.join(reportdir, 'stderr.txt')
  jsrv_html = os.path.join(reportdir, html)
  print >>ostream, 'Qt browser:         ' + jsrview_exe
  with open(jsrv_stdo, 'w') as stdo, open(jsrv_stde, 'w') as stde:
    SP.Popen((jsrview_exe, jsrv_html), cwd=reportdir, stdout=stdo, stderr=stde)

def main():
  parser = argparse.ArgumentParser(
     description='Test run of log parsers.'
  )
  parser.add_argument(
     '-q', '--quiet',
     help='launch jsrview and display results',
     action='store_true'
  )
  parser.add_argument(
     '-p', '--pause',
     help='pause in seconds',
     default=0.0,
     type=float,
     metavar='<float>'
  )
  opt = parser.parse_args()
  delay1 = opt.pause
  delay2 = opt.pause
  delay3 = opt.pause
  ccp4 = os.environ['CCP4']
  browser = os.path.join(ccp4, 'libexec', 'jsrview')
  dir_data = os.path.join(ccp4, 'examples', 'toxd')
  pdb_inp = os.path.join(dir_data, 'toxd.pdb')
  mtz_inp = os.path.join(dir_data, 'toxd_mapcoefs.mtz')
  demodir = os.path.dirname(os.path.abspath(__file__))
  testdir = os.path.join(demodir, 'testdir_jsrview')
  logfile = os.path.join(testdir, 'output.log')
  xmlfile = os.path.join(testdir, 'output.xml')

  print 'PyRVAPI Demo'
  d = API.document.newdoc(
    wintitle='RVAPI Demo 1',
    xml=xmlfile,
    ostream=sys.stdout,
    reportdir=testdir,
  )
  d.add_header('RVAPI Demo 1')
  API.flush()
  if not opt.quiet:
    start_jsrview(testdir, ccp4)

  # ****************************************************************
  #   DEMO 1:  MAKING TABS
  # ****************************************************************
  #
  # Add watched (updatable) content to the log tab. Note that the
  # log file does not exist yet.
  #
  # Flush will push incremental changes into browser
  #

  time.sleep(delay1)
  d.add_header('RVAPI Demo Page 1')

  tab2 = API.pyrvapi_tab('Log file', open=False)
  tab1 = API.pyrvapi_tab('Report', before=tab2)

  API.pyrvapi_content(tab2, logfile)

  API.flush()

  # ****************************************************************
  #   DEMO 2:  MAKING SECTIONS IN TABS
  # ****************************************************************
  #

  time.sleep(delay1)
  sec1 = API.pyrvapi_section(tab1, 'Results', 0, 0, 1, 1)
  sec2 = API.pyrvapi_section(tab1, 'Output files', 1, 0, 1, 1)
  API.flush()

  log = logWriter(logfile, 1000)

  # ****************************************************************
  #   DEMO 3:  PUTTING TEXTS INTO SECTIONS
  # ****************************************************************
  #
  # Note that one can put text straight into tab grids just as well,
  # by specifying tab ids instead of section ids in the following
  # functions
  #

  time.sleep(delay1)
  log.dump_fraction(0.035)

  sec1.add_text('Something about results.', 0, 0, 1, 1)

  txt = 'This test is only for demonstration.\n<br>'
  txt += 'This text contains a newline symbol'
  sec2.add_text(txt, 0, 0, 1, 1)

  API.flush()

  # ****************************************************************
  #   DEMO 3:  MAKING TABLES
  # ****************************************************************
  #
  # We put this table into a section, however it can be put anywhere
  # by just specifying an appropriate node Id
  #
  # Make column headers (optional)
  #
  # Make row headers (optional)
  #
  # Fill table body. Any strings may be put in, however, there are
  # interface functions for other types of data. Note that one
  # can leave empty cells and intersperse data types at will
  #

  time.sleep(delay1)
  log.dump_fraction(0.071)

  table1 = API.pyrvapi_table(sec1, 1, 0, 1, 1, 'Demo Table 1', 1)

  table1.col_title(0, text='Column 1', tooltip='Tooltip 1')
  table1.col_title(1, text='Column 2')
  table1.col_title(2, text='Column 3', tooltip='Tooltip 2')
  table1.col_title(3, text='Column 4')

  table1.row_title(0, text='Row 1', tooltip='Tooltip 1')
  table1.row_title(1, text='** Row 2 **')

  for itr in range(0,2):
    for itc in range(0,4):
      S = '%10.3g' %( float(itr+1)*float(itc+1)/3.14159265 )
      table1.cell_data(itr, itc, value=S)

  API.flush()

  table2 = API.pyrvapi_table(sec1, 2, 0, 1, 1, 'Header', 100)

  table2.col_title(0, text='Column 1', tooltip='Tooltip 1')
  table2.col_title(1, text='Column 2')
  table2.col_title(2, text='Column 3', tooltip='Tooltip 2')
  table2.col_title(3, text='Column 4')

  table2.row_title(0, text='Row 1', tooltip='Tooltip 1')
  table2.row_title(1, text='** Row 2 **')

  for itr in range(0,2):
    for itc in range(0,4):
      S = '%10.3g' %( float(itr+1)*float(itc+1)/3.14159265 )
      table2.cell_data(itr, itc, value=S)

  table2.cell_shape(0, 3, 1, 1, style='width:100%;')

  API.flush()

  # ****************************************************************
  #   DEMO 4:  MAKING DATA WIDGETS
  # ****************************************************************
  #

  time.sleep(delay1)
  log.dump_fraction(0.05)

  structure = API.pyrvapi_data('Structure and electron density', sec2, 2, 0, 1, 1, 1)
  structure.append(pdb_inp, 'xyz')
  structure.draw()
  structure.append(mtz_inp, 'hkl:map')

  API.flush()

  # ****************************************************************
  #   DEMO 5:  APPENDING, RATHER THAN SETTING WIDGETS ON THE PAGE
  # ****************************************************************
  #
  # In "append" mode, the row grid counter autoincrements, and only
  # 0th column is used for placing new element in the page
  #

  time.sleep(delay2)
  log.dump_fraction(0.09)

  txt = 'This demonstrates the \'append\' style of API functions.<br>&nbsp;'
  sec1.add_text(txt)

  sec1.set_state(False)

  # ****************************************************************
  #   DEMO 5:  MAKING A GRAPH WIDGET
  # ****************************************************************
  #
  # We choose to put a graph widget into 1st section, although it may
  # be placed anywhere
  #
  # Graph widgets contain lists of data and lists of plots. First, we
  # describe a single data block, containing x-points and 4 functions
  # on them. Note that any ids for X- and Y- values may be chosen, and
  # that they are local to the data block.
  #
  # Second, we describe individual plots, which will be switchable in
  # the right-hand side of the graph widget.
  #

  time.sleep(delay2)
  log.dump_fraction(0.03)

  graphWidget1 = API.loggraph(sec1)

  data1 = API.graph_data(graphWidget1, 'Trigonometry')
  d1x = API.graph_dataset(data1, 'x', 'argument')
  d1y1 = API.graph_dataset(data1, 'sin(x)', 'Sine', isint=False)
  d1y2 = API.graph_dataset(data1, 'cos(x)', 'Cosine', isint=False)
  d1y3 = API.graph_dataset(data1, 'sin(x)/x', 'Damped sine', isint=False)
  d1y4 = API.graph_dataset(data1, 'cos(x)/x', 'Damped cosine', isint=False)

  for i in range(1, 21):
    d1x.add_datum(i)
    d1y1.add_datum(math.sin(i* 6.28/ 19.0))
    d1y2.add_datum(math.cos(i* 6.28/ 19.0))
    d1y3.add_datum(math.sin(i* 6.28/ 19.0)/ i)
    d1y4.add_datum(math.cos(i* 6.28/ 19.0)/ i)

  plot1 = API.graph_plot(graphWidget1, 'Sine and Cosine', 'Argument', 'Functions: Sine and Cosine')
  plot1.set_xmin(-1.0)
  d1x_y1 = API.plot_line(plot1, data1, d1x, d1y1)
  d1x_y2 = API.plot_line(plot1, data1, d1x, d1y2)
  d1x_y1.set_line_fill(True, True, API.COLOR_LightCoral, 0.2)
  d1x_y2.set_line_fill(True, True, API.COLOR_Aquamarine, 0.2)

  plot2 = API.graph_plot(graphWidget1, 'Damped sine and cosine', 'Argument', 'Functions: Damped Sine and Cosine')
  plot2.set_xmax(50.0)
  d1x_y3 = API.plot_line(plot2, data1, d1x, d1y3)
  d1x_y4 = API.plot_line(plot2, data1, d1x, d1y4)

  API.flush()

  # ****************************************************************
  #   DEMO 6:  ADDING DATA TO GRAPH WIDGET - 1
  # ****************************************************************
  #
  # Adding data is quite straightforward. In this example, we just
  # create another data block, fill it with data and describe the
  # corresponding plots by just repeating the same steps as in Demo 5.
  #
  # Here we describe new plots #3 and #4, placing them under the title
  # of datablock 'data2'. Note, thoughm that they can be placed under
  # the title of datablock 'data1' just as well. Note also, that plots
  # can use data from different datablocks. Note finally, that
  # datablock may be empty and used only for providing a collective
  # title for a series of plots.
  #

  time.sleep(delay2)
  log.dump_fraction(0.065)

  data2 = API.graph_data(graphWidget1, 'Powers')
  d2x = API.graph_dataset(data2, 'x', 'argument', isint=False)
  d2y1 = API.graph_dataset(data2, 'x^2', 'Direct square', isint=False)
  d2y2 = API.graph_dataset(data2, 'x^3', 'Direct power 3', isint=False)
  d2y3 = API.graph_dataset(data2, 'x^{-2}', 'Inverse square', isint=False)
  d2y4 = API.graph_dataset(data2, 'x^{-3}', 'Inverse power 3', isint=False)

  for i in range(1, 21):
    x = (i - 10.5)/ 10.0
    d2x.add_datum(x)
    d2y1.add_datum(x* x)
    d2y2.add_datum(x* x* x)
    d2y3.add_datum(0.99999995/ (x* x))
    d2y4.add_datum(0.1/ (x* x* x))

  plot3 = API.graph_plot(graphWidget1, 'Direct powers', 'Argument', 'Functions: Powers x^2 and x^3')
  d2x_y1 = API.plot_line(plot3, data2, d2x, d2y1)
  d2x_y2 = API.plot_line(plot3, data2, d2x, d2y2)

  plot4 = API.graph_plot(graphWidget1, 'Inverse powers', 'Argument', 'Functions: Powers x^{-2} and x^{-3}')
  d2x_y3 = API.plot_line(plot4, data2, d2x, d2y3)
  d2x_y4 = API.plot_line(plot4, data2, d2x, d2y4)

  API.flush()

  # ****************************************************************
  #   DEMO 7:  ADDING DATA TO GRAPH WIDGET - 2
  # ****************************************************************
  #
  # In this example, we add XY points to the existing data block.
  # All relevant plots will update automatically in the browser as
  # soon as content is flushed.
  #

  time.sleep(delay2)
  log.dump_fraction(0.085)

  for i in range(21, 41):
    d1x.add_datum(i)
    d1y1.add_datum(math.sin(i* 6.28/ 19.0))
    d1y2.add_datum(math.cos(i* 6.28/ 19.0))
    d1y3.add_datum(math.sin(i* 6.28/ 19.0)/ i)
    d1y4.add_datum(math.cos(i* 6.28/ 19.0)/ i)

  API.flush()

  # ****************************************************************
  #   DEMO 8:  ADDING DATA TO GRAPH WIDGET - 3
  # ****************************************************************
  #
  # In this example, we add both data to a data block and a new plot
  # to the list of plots.
  #
  # Note that X-points in the selected data block already run from
  # 1 to 40, from previous updates. However, it is Ok to provide new
  # Y-values only for a subset of them, e.g, in range of 1-20.
  #

  time.sleep(delay2)
  log.dump_fraction(0.025)

  d1y5 = API.graph_dataset(data1, 'x*sin(x)', 'Amplified sine' , isint=False)
  d1y6 = API.graph_dataset(data1, 'x*cos(x)', 'Amplified cosine' , isint=False)

  for i in range(1, 21):
    d1y5.add_datum(i* math.sin(i* 6.28/ 19.0))
    d1y6.add_datum(i* math.cos(i* 6.28/ 19.0))

  plot5 = API.graph_plot(graphWidget1, 'Amplified sine and cosine', 'Argument', 'Functions: Amplified Sine and Cosine')
  d1x_y5 = API.plot_line(plot5, data1, d1x, d1y5)
  d1x_y6 = API.plot_line(plot5, data1, d1x, d1y6)

  for i in range(-2, 25, 2):
    s = '[%d]' %((22 - i)* (22 - i))
    plot5.add_xtick(i, s)

  API.flush()

  # ****************************************************************
  #   DEMO 9:  MAKING TREE WIDGETS
  # ****************************************************************
  #

  time.sleep(delay2)
  log.dump_fraction(0.095)

  tree_tab = API.pyrvapi_tab('Tree Widget')
  tree1 = API.tree_widget(tree_tab, 'Tree Example', 0, 0, 1, 1)
  tree_panel1 = API.tree_node(tree1, 'Node 1', 'auto')
  tree_panel2 = API.tree_node(tree1, 'Node 2', 'open')
  tree_panel21 = API.tree_node(tree_panel2, 'Node 21 make this long', 'auto')
  tree_panel22 = API.tree_node(tree_panel2, 'Node 22', 'auto')

  tree_sec1 = API.pyrvapi_section(tree_panel1, 'Section for NODE 1', 0, 0, 1, 1)
  tree_sec2 = API.pyrvapi_section(tree_panel2, 'Section for NODE 2', 0, 0, 1, 1)
  tree_sec21 = API.pyrvapi_section(tree_panel21, 'Section for NODE 21', 0, 0, 1, 1)
  tree_sec22 = API.pyrvapi_section(tree_panel22, 'Section for NODE 22', 0, 0, 1, 1)

  tree_sec1.add_text('Text for NODE 1', 0, 0, 1, 1)
  tree_sec2.add_text('Text for NODE 2', 0, 0, 1, 1)
  tree_sec21.add_text('Text for NODE 21', 0, 0, 1, 1)
  tree_sec22.add_text('Text for NODE 22', 0, 0, 1, 1)

  API.flush()

  time.sleep(delay2)

  tree_sec1.add_text('Another Text for NODE 1')

  API.flush()

  # ****************************************************************
  #   DEMO 10:  MAKING RADAR WIDGETS
  # ****************************************************************
  #

  time.sleep(delay2)
  log.dump_fraction(0.1)

  radar1 = API.radar(tree_tab, 'Radar example', 1, 0, 1, 1, 0)
  radar1.add_property('prop 1', 0.1)
  radar1.add_property('prop 2', 0.5)
  radar1.add_property('prop 3', 0.3)
  radar1.add_property('prop 4', 0.9)
  radar1.add_property('prop 5', 1.0)
  radar1.add_property('prop 6', 0.7)

  API.flush()

  # ****************************************************************
  #   DEMO 11:  UPDATING GRAPHS IN TREE LEAFS
  # ****************************************************************
  #
  # We choose to put a graph widget into 1st section, although it may
  # be placed anywhere
  #
  # Graph widgets contain lists of data and lists of plots. First, we
  # describe a single data block, containing x-points and 4 functions
  # on them. Note that any ids for X- and Y- values may be chosen, and
  # that they are local to the data block.
  #
  # Second, we describe individual plots, which will be switchable in
  # the right-hand side of the graph widget.
  #
  # Finally, push changes into the browser.
  #

  time.sleep(delay3)
  log.dump_fraction(0.105)

  tree_graph_tab = API.pyrvapi_tab('Tree Widget with Graphs')
  tree2 = API.tree_widget(tree_graph_tab, 'Tree Graph Example', 0, 0, 1, 1)
  tree2_panel1 = API.tree_node(tree2, 'Graph 1', 'auto')

  graphWidget2 = API.loggraph(tree2_panel1, 0, 0, 1, 1)

  data21 = API.graph_data(graphWidget2, 'Trigonometry')
  d21x = API.graph_dataset(data21, 'x', 'argument')
  d21y1 = API.graph_dataset(data21, 'sin(x)', 'Sine', isint=False)
  d21y2 = API.graph_dataset(data21, 'cos(x)', 'Cosine', isint=False)
  d21y3 = API.graph_dataset(data21, 'sin(x)/x', 'Damped sine', isint=False)
  d21y4 = API.graph_dataset(data21, 'cos(x)/x', 'Damped cosine', isint=False)

  for i in range(1, 21):
    d21x.add_datum(i)
    d21y1.add_datum(math.sin(i* 6.28/ 19.0))
    d21y2.add_datum(math.cos(i* 6.28/ 19.0))
    d21y3.add_datum(math.sin(i* 6.28/ 19.0)/ i)
    d21y4.add_datum(math.cos(i* 6.28/ 19.0)/ i)

  plot21 = API.graph_plot(graphWidget2, 'Sine and Cosine', 'Argument', 'Functions: Sine and Cosine')
  d21x_y1 = API.plot_line(plot21, data21, d21x, d21y1)
  d21x_y2 = API.plot_line(plot21, data21, d21x, d21y2)
  d21x_y1.set_options(color=API.COLOR_IndianRed, style=API.LINE_Bars)
  d21x_y2.set_options(color=API.COLOR_Gold, style=API.LINE_Off)
  d21x_y1.set_options(marker=API.MARKER_filledCircle, width=1.0)
  d21x_y2.set_options(marker=API.MARKER_filledCircle, width=2.5)

  plot22 = API.graph_plot(graphWidget2, 'Damped sine and cosine', 'Argument', 'Functions: Damped Sine and Cosine')
  plot22.set_xrange(0.0, 40.0)
  plot22.set_yrange(-0.4, 1.2)
  d21x_y3 = API.plot_line(plot22, data21, d21x, d21y3)
  d21x_y4 = API.plot_line(plot22, data21, d21x, d21y4)

  API.flush()

  # ****************************************************************
  #   DEMO 12:  ADDING DATA TO GRAPH WIDGET IN TREE LEAF
  # ****************************************************************
  #
  # Adding data is quite straightforward. In this example, we just
  # create another data block, fill it with data and describe the
  # corresponding plots by just repeating the same steps as in Demo 5.
  #
  # Here we describe new plots #23 and #24, placing them under the title
  # of datablock 'data2'. Note, thoughm that they can be placed under
  # the title of datablock 'data1' just as well. Note also, that plots
  # can use data from different datablocks. Note finally, that
  # datablock may be empty and used only for providing a collective
  # title for a series of plots.
  #

  time.sleep(delay3)
  log.dump_fraction(0.11)

  data22 = API.graph_data(graphWidget2, 'Powers')
  d22x = API.graph_dataset(data22, 'x', 'argument', isint=False)
  d22y1 = API.graph_dataset(data22, 'x^2', 'Direct square', isint=False)
  d22y2 = API.graph_dataset(data22, 'x^3', 'Direct power 3', isint=False)
  d22y3 = API.graph_dataset(data22, 'x^{-2}', 'Inverse square', isint=False)
  d22y4 = API.graph_dataset(data22, 'x^{-3}', 'Inverse power 3', isint=False)

  for i in range(1, 21):
    x = (i - 10.5)/ 10.0
    d22x.add_datum(x)
    d22y1.add_datum(x* x)
    d22y2.add_datum(x* x* x)
    d22y3.add_datum(0.99999995/ (x* x))
    d22y4.add_datum(0.1/ (x* x* x))

  plot23 = API.graph_plot(graphWidget2, 'Direct powers', 'Argument', 'Functions: Powers x^2 and x^3')
  d22x_y1 = API.plot_line(plot23, data22, d22x, d22y1)
  d22x_y2 = API.plot_line(plot23, data22, d22x, d22y2)

  plot24 = API.graph_plot(graphWidget2, 'Inverse powers', 'Argument', 'Functions: Powers x^{-2} and x^{-3}')
  d22x_y3 = API.plot_line(plot24, data22, d22x, d22y3)
  d22x_y4 = API.plot_line(plot24, data22, d22x, d22y4)

  API.flush()

  # ****************************************************************
  #   DEMO 13:  ADDING DATA TO GRAPH WIDGET IN TREE LEAF - 2
  # ****************************************************************
  #
  # In this example, we add XY points to the existing data block.
  # All relevant plots will update automatically in the browser as
  # soon as content is flushed.
  #

  time.sleep(delay3)
  log.dump_fraction(0.115)

  for i in range(21, 41):
    d21x.add_datum(i)
    d21y1.add_datum(math.sin(i* 6.28/ 19.0))
    d21y2.add_datum(math.cos(i* 6.28/ 19.0))
    d21y3.add_datum(math.sin(i* 6.28/ 19.0)/ i)
    d21y4.add_datum(math.cos(i* 6.28/ 19.0)/ i)
    time.sleep(delay3)
    API.flush()

  # ****************************************************************
  #   DEMO 14:  ADDING DATA TO GRAPH WIDGET IN TREE LEAF - 3
  # ****************************************************************
  #
  # In this example, we add both data to a data block and a new plot
  # to the list of plots.
  #
  # Note that X-points in the selected data block already run from
  # 1 to 40, from previous updates. However, it is Ok to provide new
  # Y-values only for a subset of them, e.g, in range of 1-20.
  #

  time.sleep(delay3)
  log.dump_fraction(0.025)

  d21y5 = API.graph_dataset(data21, 'x*sin(x)', 'Amplified sine', isint=False)
  d21y6 = API.graph_dataset(data21, 'x*cos(x)', 'Amplified cosine', isint=False)

  for i in range(1, 21):
    d21y5.add_datum(i* math.sin(i* 6.28/ 19.0))
    d21y6.add_datum(i* math.cos(i* 6.28/ 19.0))

  plot25 = API.graph_plot(graphWidget2, 'Amplified sine and cosine', 'Argument', 'Functions: Amplified Sine and Cosine' )
  d21x_y5 = API.plot_line(plot25, data21, d21x, d21y5)
  d21x_y6 = API.plot_line(plot25, data21, d21x, d21y6)

  API.flush()

  # ****************************************************************
  #   DEMO 14:  ADDING MORE TREE NODES
  # ****************************************************************
  #

  time.sleep(delay3)
  log.dump_fraction(0.12)

  tree2_panel2 = API.tree_node(tree2, 'Text 2', 'auto')

  tree2_panel2.add_text('JUST A PLLACEHOLDER FOR TREE LEAF', 0, 0, 1, 1)

  API.flush()

  # ****************************************************************
  #   END
  # ****************************************************************

  log.dump_fraction(1.0)
  log.ostream.close()

if __name__ == '__main__':
  main()


