#
#     Copyright (C) 2016 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#


import os
import json
from ccpem_core.ccpem_utils import ccpem_argparser
from ccpem_core import ccpem_utils
from ccpem_core import process_manager
from ccpem_core.tasks import task_utils
from ccpem_core.tasks.tempy.scores import scores_process
from ccpem_core.process_manager import job_register
from ccpem_core.tasks.ribfind import ribfind_task
from ccpem_core import settings

class GlobScore(task_utils.CCPEMTask):
    '''
    CCPEM / TEMPy global fit scores wrapper.
    '''
    task_info = task_utils.CCPEMTaskInfo(
        name='TEMPyGlobScore',
        author='A. Joseph, M.Topf, M. Winn',
        version='1.1',
        description=(
            '''Global scoring using TEMPy library.<br>
            Input a map and one or more atomic models.<br>
            Multiple scores are calculated to evaluate atomic model
            fit. The 'local_*' scores are calculated on the voxels 
            in the region of overlap between contoured map and model.
            These scores are combined with the extent of overlap 
            between the map and model to estimate combined scores 
            (ccc_ov,mi_ov).<br>
            Please see the references for the details of the 
            scores:<br>
            <a href=https://www.ncbi.nlm.nih.gov/pubmed/28552721> 
            Joseph et al. 2017</a><br>
            <a href="https://www.ncbi.nlm.nih.gov/pubmed/26306092">
            Farabella et al. 2015</a>
            '''),
        short_description=(
            '[BETA TEST] Multiple global scores to evaluate atomic model'
            ' fit'),
        documentation_link='http://topf-group.ismb.lon.ac.uk/TEMPY.html',
        references=('https://www.ncbi.nlm.nih.gov/pubmed/28552721',
                    'https://www.ncbi.nlm.nih.gov/pubmed/26306092'))

    commands = {'ccpem-python':
        ['ccpem-python', os.path.realpath(scores_process.__file__)]}


    def __init__(self,
                 database_path=None,
                 args=None,
                 args_json=None,
                 pipeline=None,
                 job_location=None,
                 parent=None):
        #
        super(GlobScore, self).__init__(
            database_path=database_path,
            args=args,
            args_json=args_json,
            pipeline=pipeline,
            job_location=job_location,
            parent=parent)

    def parser(self):
        parser = ccpem_argparser.ccpemArgParser()
        #
        job_title = parser.add_argument_group()
        job_title.add_argument(
            '-job_title',
            '--job_title',
            help='Short description of job',
            metavar='Job title',
            type=str,
            default=None)
        #
        job_location = parser.add_argument_group()
        job_location.add_argument(
            '-job_location',
            '--job_location',
            help='Directory to run job',
            metavar='Job location',
            type=str,
            default=None)
        #
        map_path = parser.add_argument_group()
        map_path.add_argument(
            '-map_path',
            '--map_path',
            help='Input map (mrc format)',
            metavar='Input map',
            type=str,
            default=None)
        #
        parser.add_argument(
            '-input_pdbs',
            '--input_pdbs',
            help='One or more models to score against map',
            metavar='Input PDB',
            type=str,
            nargs='*',
            default=None)
        #
        parser.add_argument(
            '-input_pdb_chains',
            '--input_pdb_chains',
            help='Input PDB chain(s)',
            metavar='Input PDB chain selections',
            type=str,
            nargs='*',
            default=None)
        #
        parser.add_argument(
            '-map_resolution',
            '--map_resolution',
            help=('Resolution of input map (Angstrom)'),
            metavar='Map resolution',
            type=float,
            default=None)
        
        parser.add_argument(
            '-auto_contour_level',
            '--auto_contour_level',
            help='''Automatically set map contour level. Unselect to set manually.''',
            metavar='Auto set contour',
            type=bool,
            default=True)
        #
        parser.add_argument(
            '-map_contour_level',
            '--map_contour_level',
            help='''Set map contour level''',
            metavar='Contour level',
            type=float,
            default=0.0)

        #
        return parser

    def edit_argsfile(self,args_file):
        with open(args_file,'r') as f:
            json_args = json.load(f)
            #set mapprocess task output
            if self.mapprocess_task is not None:
                mapprocess_job_location = self.mapprocess_task.job_location
                #the processed map is written to the same dir as input map
                #the job location has other output files
                if mapprocess_job_location is not None:
                    processed_map_path = os.path.splitext(
                                        os.path.abspath(self.args.map_path.value))[0] \
                                        +'_processed.mrc'
                    #print processed_map_path
                    if os.path.isfile(processed_map_path):
                        json_args['map_path'] = processed_map_path
                        self.unprocessed_map = self.args.map_path.value
                        self.args.map_path.value = processed_map_path
        with open(args_file,'w') as f:
            json.dump(json_args,f)

    def check_processed_map(self):
        processed_map_path = os.path.splitext(
                            os.path.abspath(self.args.map_path.value))[0] \
                            +'_processed.mrc'
        #print processed_map_path
        if os.path.isfile(processed_map_path) and \
            self.mapprocess_task is not None:
            self.map_input.value_line.setText(processed_map_path)

    def run_pipeline(self, job_id=None, db_inject=None):
        
        pl = None
        # Set args
        args_file = os.path.join(self.job_location,
                                 'args.json')
        #check for mapprocess output
        self.check_processed_map()
        self.edit_argsfile(args_file)
        # Generate process
        self.score_wrapper = GlobScoreWrapper(
            command=self.commands['ccpem-python'],
            job_location=self.job_location,
            name='TEMPy scores')

        pl = [[self.score_wrapper.process]]
        # pipeline
        self.pipeline = process_manager.CCPEMPipeline(
            pipeline=pl,
            job_id=job_id,
            args_path=self.args.jsonfile,
            location=self.job_location,
            database_path=self.database_path,
            db_inject=db_inject,
            taskname=self.task_info.name,
            title=self.args.job_title.value)
        self.pipeline.start()

class GlobScoreWrapper(object):
    '''
    Wrapper for TEMPy GlobScore process.
    '''
    def __init__(self,
                 command,
                 job_location,
                 name=None):
        self.job_location = ccpem_utils.get_path_abs(job_location)
        self.name = name
        if self.name is None:
            self.name = self.__class__.__name__

        # Set args
        self.args = os.path.join(self.job_location,
                                 'args.json')
        
        # Set process
        assert command is not None
        self.process = process_manager.CCPEMProcess(
            name=self.name,
            command=command,
            args=self.args,
            location=self.job_location,
            stdin=None)
