
import os, sys, shutil, time
import subprocess as SP
import pyrvapi_ext as API
import regex_tree as RT
from xml.etree import ElementTree as ET

class arpwarp_rvapi_calls(object):

    def flush(self):
        if self.flush_pause > 0:
            time.sleep(self.flush_pause)

        API.flush()

    def __init__(self, job_params, rvapi_grid, flush_pause, resfile=None):
        self.flush_pause = flush_pause
        self.resfile = resfile
        self.residues = int(job_params.get('residues', 0))
        buildingcycles = int(job_params['buildingcycles'])
        restrref = int(job_params['restrref'])
        multit = int(job_params['multit'])
        self.building_cycle = 0
        plt_ymax = bool(int(job_params.get('plt_ymax', '0')))

        sec = API.pyrvapi_section(rvapi_grid, 'Results', 0, 0, 1, 1)

        self.table = API.pyrvapi_table(sec, 0, 0, 1, 1, 'Summary', 0)
        self.table.row_title(0, 'Residues built', 'No of residues built')
        self.table.row_title(1, 'Chains built', 'No of continuous fragments')
        self.table.row_title(2, 'Longest chain', 'No of continuous in the longest fragment')
        self.table.row_title(3, 'R-cryst', 'Crystallographic R-factor')
        self.table.row_title(4, 'R-free', 'Free R-factor')
        self.table.body_cell(0, 0, 'N/A')
        self.table.body_cell(1, 0, 'N/A')
        self.table.body_cell(2, 0, 'N/A')
        self.table.body_cell(3, 0, 'N/A')
        self.table.body_cell(4, 0, 'N/A')
        self.flush()

        gwd = API.loggraph(sec, 1, 0, 1, 1)

        gdt = API.graph_data(gwd, 'Refinements')
        self.d1x = API.graph_dataset(gdt, 'x', 'argument')
        self.d1y1 = API.graph_dataset(gdt, 'R-cryst', 'Crystallographic R-factor', False)
        self.d1y2 = API.graph_dataset(gdt, 'R-free', 'Crystallographic free R-factor', False)

        plt = API.graph_plot(gwd, 'R-factors', 'Refinement cycle', 'R')
        plt.set_xrange(0, buildingcycles* restrref - 1)
        if plt_ymax:
            plt.set_yrange(-0.02, 0.6)

        else:
            plt.set_ymin(-0.02)

        pltline = API.plot_line(plt, gdt, self.d1x, self.d1y1)
        pltline = API.plot_line(plt, gdt, self.d1x, self.d1y2)

        gdt = API.graph_data(gwd, 'Building')
        self.d2x = API.graph_dataset(gdt, 'x', 'argument')
        self.d2y1 = API.graph_dataset(gdt, 'in model', 'Crystallographic R-factor', False)
        self.d2y2 = API.graph_dataset(gdt, 'in longest chain', 'Crystallographic free R-factor', False)

        plt = API.graph_plot(gwd, 'Residues built', 'Building round', 'Residues')
        plt.set_xrange(1, buildingcycles* multit)
        if plt_ymax and self.residues:
            plt.set_yrange(0, self.residues)

        else:
            plt.set_ymin(0)

        pltline = API.plot_line(plt, gdt, self.d2x, self.d2y1)
        pltline = API.plot_line(plt, gdt, self.d2x, self.d2y2)

        self.flush()

        self.chains = 0
        self.r_free = 0
        self.r_cryst = 0

    def cycle_ready(self, items):
        pass

    def cycle_started(self, items):
        pass

    def building_round_finished(self, items):
        self.building_cycle += 1
        round_no, in_model, chains, in_longest, score = items
        self.d2x.add_datum(self.building_cycle)
        self.d2y1.add_datum(float(in_model))
        self.d2y2.add_datum(float(in_longest))

        self.table.body_cell(0, 0, in_model)
        self.table.body_cell(1, 0, chains)
        self.table.body_cell(2, 0, in_longest)
        self.flush()

        self.frac_built = -1
        if self.residues:
            self.frac_built = float(in_model)/ self.residues

    def building_step_finished(self, items):
        pass

    def refinement_step_finished(self, items):
        self.d1x.add_datum(int(items[0]))
        self.d1y1.add_datum(float(items[1]))
        self.d1y2.add_datum(float(items[2]))

        self.table.body_cell(3, 0, items[1])
        self.table.body_cell(4, 0, items[2])
        self.flush()

        self.r_cryst = items[1]
        self.r_free = items[2]

    def cycle_finished(self, items):
        pass

    def job_finished(self, items):
        if self.resfile and os.path.isdir(os.path.dirname(self.resfile)):
            items = str(self.frac_built), str(self.r_free), str(self.r_cryst), ''
            with open(self.resfile, 'w') as ostream:
              ostream.write('\n'.join(items))

class arpwarp_log_parser(RT.LogDataParser):

    def __init__(self):
        RT.LogDataParser.__init__(self)

        version_re = ' *You are running ARP/wARP version ([0-9]*\.[0-9]*)'
        big_cycle_re = ' Building Cycle ([0-9]*)    Atomic shape factors'
        big_cycle_re += '  ([0-9]*\.[0-9]*)  ([0-9]*\.[0-9]*)\n'
        building_round_re = '   Round ([0-9]*): ([0-9]*) peptides, ([0-9]*) chains.'
        building_round_re += ' Longest chain ([0-9]*) peptides. Score ([0-9]*\.[0-9]*)\n'
        building_re = '  Chains ([0-9]*), Residues ([0-9]*),'
        building_re += ' Estimated correctness of the model ([0-9]*\.[0-9]*) %\n'
        refinement_re = '  Cycle ([0-9]*): *After refmac, R = ([0-9]*\.[0-9]*)'
        refinement_re += ' \(Rfree = ([0-9]*\.[0-9]*)\) for ([0-9]*) atoms.\n'
        success_re = ' Normal termination of warpNtrace'

        self.version = RT.LogDataLine(version_re)
        self.big_cycle = RT.LogDataLine(big_cycle_re)
        self.building_round = RT.LogDataLine(building_round_re)
        self.building = RT.LogDataLine(building_re)
        self.refinement = RT.LogDataLine(refinement_re)
        self.success = RT.LogDataLine(success_re)

        self.add_next(self.version)
        self.version.add_next(self.refinement)
        self.big_cycle.add_next(self.building_round)
        self.building_round.add_next(self.building_round, self.building)
        self.building.add_next(self.refinement)
        self.refinement.add_next(self.refinement, self.big_cycle, self.success)

    def add_listener(self, listener):
        self.version.add_action(listener.cycle_ready)
        self.big_cycle.add_action(listener.cycle_finished, listener.cycle_ready, listener.cycle_started)
        self.building_round.add_action(listener.building_round_finished)
        self.building.add_action(listener.building_step_finished)
        self.refinement.add_action(listener.refinement_step_finished)
        self.success.add_action(listener.cycle_finished, listener.job_finished)

def arpwarp_parser(rvapi_grid, job_params=None, flush_pause=-1, resfile=None, **kwargs):
    parser = arpwarp_log_parser()
    parser.add_listener(arpwarp_rvapi_calls(job_params, rvapi_grid, flush_pause, resfile))
    return parser

def main():
    mock = None
    action_pause = 0
    verbose = False
    ofile = None
    xmlout = None
    resfile = None
    reportdir = 'report_awa'
    cmd = ['auto_tracing.sh']
    arg_dict = {}
    stdin_lines = sys.stdin.readlines()
    for line in stdin_lines:
        if not line.startswith('#'):
            key, sep, value = line.strip().partition(' ')
            value = value.lstrip()
            if key == 'mockdir':
                mock = os.path.join(value, 'warpNtrace')

            elif key == 'pause':
                action_pause = 0.001* int(value)

            elif key == 'reportdir':
                reportdir = value

            elif key == 'xmlout':
                xmlout = value

            elif key == 'logfile':
                ofile = value

            elif key == 'resfile':
                resfile = value

            elif key == 'verbose':
                verbose = bool(value)

            else:
                arg_dict[key] = value
                cmd.append(key)
                cmd.append(value)

    if not os.path.isdir(reportdir):
        os.mkdir(reportdir)

    if ofile:
        ostream = open(ofile, 'w')

    else:
        ostream = sys.stdout

    doc = API.document.newdoc(
        wintitle = 'ARP/wARP report page',
        reportdir = reportdir,
        ostream = ostream if verbose else None,
        xml = xmlout,
    )
    doc.add_header('ARP/wARP report page')
    API.flush()
    restab = API.pyrvapi_tab('Results')
    if ofile:
        logtab = API.pyrvapi_tab('Log', open=False)
        API.pyrvapi_content(logtab, ofile)

    if verbose:
        ostream.write(''.join(stdin_lines))

    parser = arpwarp_parser(restab, arg_dict, -1, resfile)
    if mock:
        istream = open(mock + '.log')
        parser.parse_stream(open(mock + '.log'), ostream, pause=action_pause)
        istream.close()
        jobId = arg_dict['jobId']
        jobdir = os.path.join(arg_dict['workdir'], jobId)
        if not os.path.isdir(jobdir):
            os.mkdir(jobdir)

        oprefix, mtz_ext, ending = os.path.basename(arg_dict['datafile']).rpartition('.mtz')
        assert mtz_ext == '.mtz' and not ending
        opath = os.path.join(jobdir, oprefix + '_warpNtrace')
        shutil.copy(mock + '.mtz', opath + '.mtz')
        shutil.copy(mock + '.pdb', opath + '.pdb')
        retcode = 0

    else:
        p = SP.Popen(cmd, stdout=SP.PIPE)
        parser.parse_stream(p.stdout, ostream)
        p.wait()
        p.stdout.close()
        retcode = p.returncode

    if ofile:
        ostream.close()

    sys.exit(retcode)

if __name__ == '__main__':
    main()

