#
#     Copyright (C) 2019 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.

from __future__ import print_function

import argparse
import os
import gemmi
from ccpem_core import ccpem_utils

def output_cif(model, output_name):
    ccpem_utils.print_sub_header('Output CIF')
    print('Input:', model)
    structure = gemmi.read_structure(model)
    output_name = os.path.splitext(output_name)[0] + '.cif'
    structure.make_mmcif_document().write_file(output_name)
    print('Output:', output_name)
    return output_name

def output_pdb(model, output_name):
    ccpem_utils.print_sub_header('Output PDB')
    print('Input:', model)
    structure = gemmi.read_structure(model)
    output_name = os.path.splitext(output_name)[0] + '.pdb'
    structure.write_pdb(output_name)
    print('Output:', output_name)
    return output_name

def main():
    parser = argparse.ArgumentParser(description='CCP-EM model tools')
    parser.add_argument(
        '-model', '--model',
        required=True,
        help='Input atomic model file (PDB or mmCIF/PDBx')
    parser.add_argument(
        '-output_cif', '--output_cif',
        action='store_true',
        help='Export mmcif/PDBx format')
    parser.add_argument(
        '-output_pdb', '--output_pdb', 
        action='store_true',
        help='Export pdb format')
    parser.add_argument(
        '-output_name', '--output_name', dest='output_name',
        required=False, default='model', 
        help='Output file name')
    args = parser.parse_args()

    ccpem_utils.print_header('CCP-EM Model Tools')
    if args.output_cif:
        output_cif(model=args.model, output_name=args.output_name)
    if args.output_pdb:
        output_pdb(model=args.model, output_name=args.output_name)

if __name__ == '__main__':
    main()
