#
#     Copyright (C) 2016 CCP-EM
#
#     This code is distributed under the terms and conditions of the
#     CCP-EM Program Suite Licence Agreement as a CCP-EM Application.
#     A copy of the CCP-EM licence can be obtained by writing to the
#     CCP-EM Secretary, RAL Laboratory, Harwell, OX11 0FA, UK.
#
import os

from PyQt4 import QtCore

from ccpem_gui.utils import window_utils
from ccpem_core.tasks.tempy.difference_map import difference_map_task,\
    difference_map_chimera
from ccpem_core.settings import which
from ccpem_gui.utils import command_line_launch
from ccpem_core.ccpem_utils import get_test_data_path
from ccpem_core.test_data.tasks.tempy import difference_map as test_data


# To Do
# Add spectra plot
# Options to add - to be uncomment below when Agnel adds command line arg:
#    -> Model vs map
#    -> Map alignment option (slow)

class DifferenceMapWindow(window_utils.CCPEMTaskWindow):
    '''
    TEMPy Difference map window.
    '''
    gui_test_args = get_test_data_path(test_data, 'unittest_args.json')

    def __init__(self,
                 task,
                 parent=None):
        super(DifferenceMapWindow, self).__init__(task=task,
                                                  parent=parent)
        self.output_pdb = None

    def set_args(self):
        '''
        Set input arguments
        '''
        # Job title
        self.title_input = window_utils.TitleArgInput(
            parent=self,
            arg_name='job_title',
            args=self.args)
        self.args_widget.args_layout.addWidget(self.title_input)
        self.title_input.value_line.editingFinished.connect(
            self.handle_title_set)

        # Input map 1
        self.map_input_1 = window_utils.FileArgInput(
            parent=self,
            arg_name='map_path_1',
            required=True,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.map_input_1)
        # Map 1 resolution
        self.map_resolution_1 = window_utils.NumberArgInput(
            parent=self,
            arg_name='map_resolution_1',
            required=True,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.map_resolution_1)

        # Input map or PDB
        self.map_or_pdb = window_utils.ChoiceArgInput(
            parent=self,
            arg_name='map_or_pdb_selection',
            required=False,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.map_or_pdb)
        self.map_or_pdb.value_line.currentIndexChanged.connect(
            self.set_pdb_or_map_input)

        # Input map 2
        self.map_input_2 = window_utils.FileArgInput(
            parent=self,
            arg_name='map_path_2',
            required=False,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.map_input_2)
        # Map 2 resolution
        self.map_resolution_2 = window_utils.NumberArgInput(
            parent=self,
            arg_name='map_resolution_2',
            required=False,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.map_resolution_2)
        # Input pdb
        self.pdb_input = window_utils.FileArgInput(
            parent=self,
            arg_name='pdb_path',
            required=False,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.pdb_input)
        
        # global or local mode
        self.global_or_local = window_utils.ChoiceArgInput(
            parent=self,
            arg_name='mode_selection',
            required=False,
            args=self.args)
        self.args_widget.args_layout.addWidget(self.global_or_local)
        self.global_or_local.value_line.currentIndexChanged.connect(
            self.set_mode_options)
        
        self.local_options_frame = window_utils.CCPEMExtensionFrame(
            button_name='Local mode options',
            button_tooltip='Options specific for local mode')
        self.args_widget.args_layout.addLayout(self.local_options_frame)
        
        self.maskmap = window_utils.FileArgInput(
            parent=self,
            arg_name='maskfile',
            required=False,
            args=self.args)
        self.local_options_frame.add_extension_widget(self.maskmap)
        
#         self.use_mpi_input = window_utils.ChoiceArgInput(
#             parent=self,
#             arg_name='use_mpi',
#             required=False,
#             args=self.args)
#         self.local_options_frame.add_extension_widget(self.use_mpi_input)
#         self.use_mpi_input.value_line.currentIndexChanged.connect(
#             self.set_n_mpi_visible)
#         
#         if self.args.n_mpi.value == 1:
#             self.args.n_mpi.value = QtCore.QThread.idealThreadCount() / 2
#         self.n_mpi_input = window_utils.NumberArgInput(
#             parent=self,
#             arg_name='n_mpi',
#             required=False,
#             args=self.args)
#         self.local_options_frame.add_extension_widget(self.n_mpi_input)

        self.window_size = window_utils.NumberArgInput(
            parent=self,
            arg_name='window_size',
            required=False,
            args=self.args)
        self.local_options_frame.add_extension_widget(self.window_size)

        # Extended options
        options_frame = window_utils.CCPEMExtensionFrame(
            button_name='Extended options',
            button_tooltip='Extended difference map options')
        self.args_widget.args_layout.addLayout(options_frame)
        #add advanced options
        self.add_diffmap_advanced_args(options_frame)

#         # XXX TODO: placement holder until Agnel finishes map alignment code
#         # -> Map alignement
#         map_alignment = window_utils.ChoiceArgInput(
#             parent=self,
#             arg_name='map_alignment',
#             args=self.args)
#         options_frame.add_extension_widget(map_alignment)
        self.set_pdb_or_map_input()
        self.set_mode_options()
        self.set_dust_prob()
        #self.set_n_mpi_visible()
        #add files to launcher
        self.add_diffmap_input_files_to_launcher()


    def add_diffmap_advanced_args(self,options_frame):
        # -> Use second map as reference for scaling
        self.reference_scaling = window_utils.CheckArgInput(
            parent=self,
            arg_name='refscale',
            args=self.args)
        options_frame.add_extension_widget(self.reference_scaling)
        self.reference_scaling.value_line.stateChanged.connect(self.set_ref_scale)
        # -> Threshold for fractional difference
        self.threshold_difference = window_utils.NumberArgInput(
            parent=self,
            arg_name='threshold_fraction',
            args=self.args)
        options_frame.add_extension_widget(self.threshold_difference)
        # -> Dust filter
        self.dust_filter = window_utils.ChoiceArgInput(
            parent=self,
            arg_name='dust_filter',
            args=self.args)
        options_frame.add_extension_widget(self.dust_filter)
        self.dust_filter.value_line.currentIndexChanged.connect(
            self.set_dust_prob)
        self.dust_prob = window_utils.NumberArgInput(
            parent=self,
            arg_name='dustprob',
            args=self.args)
        options_frame.add_extension_widget(self.dust_prob)
        # -> Fractional difference
        self.frac_map = window_utils.ChoiceArgInput(
            parent=self,
            arg_name='save_fracmap',
            args=self.args)
        options_frame.add_extension_widget(self.frac_map)

    def add_diffmap_input_files_to_launcher(self):
                # Add files to launcher
        self.launcher.add_file(
            arg_name='map_path_1',
            file_type='map',
            description=self.args.map_path_1.help,
            selected=True)
        if self.args.map_or_pdb_selection.value == 'Map':
            self.launcher.add_file(
                arg_name='map_path_2',
                file_type='map',
                description=self.args.map_path_2.help,
                selected=True)
        else:
            self.launcher.add_file(
                arg_name='pdb_path',
                file_type='pdb',
                description=self.args.pdb_path.help,
                selected=True)


    def set_pdb_or_map_input(self):
        #index = self.args.map_or_pdb_selection.choices.index(
        #    self.args.map_or_pdb_selection.value)
        #sel = self.args.map_or_pdb_selection.choices[index]
        sel = self.args.map_or_pdb_selection.value
        if sel == 'Model':
            self.map_input_2.required=False
            self.map_input_2.set_required(False)
            self.map_input_2.hide()
            self.map_resolution_2.required=False
            self.map_resolution_2.set_required(False)
            self.map_resolution_2.hide()
            self.pdb_input.set_required(True)
            self.pdb_input.show()
            self.task.args.map_path_2.value = None
            self.task.args.refscale.value = True
            self.reference_scaling.value_line.setChecked(True)
            
        elif sel == 'Map':
            self.map_input_2.set_required(True)
            self.map_input_2.show()
            self.map_resolution_2.show()
            self.map_resolution_2.set_required(True)
            self.pdb_input.required=False
            self.pdb_input.set_required(False)
            self.pdb_input.hide()
            self.task.args.pdb_path.value = None
            self.task.args.refscale.value = False
            self.reference_scaling.value_line.setChecked(False)
    
    def set_ref_scale(self):
        if self.reference_scaling.value_line.isChecked():
            self.task.args.refscale.value = True
        else:
            self.task.args.refscale.value = False
    
    def set_mode_options(self):
        sel = self.args.mode_selection.value
        if sel == 'local':
            self.local_options_frame.show()
            self.maskmap.show()
            self.window_size.show()
            #self.use_mpi_input.show()
        elif sel == 'global':
            self.local_options_frame.hide()
            self.maskmap.hide()
            self.window_size.hide()
            #self.use_mpi_input.hide()
        elif sel == 'None':
            self.local_options_frame.hide()
            self.maskmap.hide()
            self.window_size.hide()
            #self.use_mpi_input.hide()
            self.task.args.noscale.value = True
    
    def set_dust_prob(self):
        if self.args.dust_filter.value:
            self.dust_prob.show()
        else:
            self.dust_prob.hide()
    
    def set_n_mpi_visible(self):
        if self.args.use_mpi():
            self.n_mpi_input.show()
        else:
            self.n_mpi_input.hide()
        

    def set_on_job_running_custom(self):
        if self.args.map_or_pdb_selection.value == 'PDB':
            self.launcher.add_file(
                arg_name='pdb_path',
                file_type='pdb',
                description=self.args.pdb_path.help,
                selected=True)

    def set_on_job_finish_custom(self):
        # Set expected output files as implemented in TEMPy/difference_map
        diff_map1, diff_map2 = self.get_difference_map_names()
        # Add output files to launcher
        if self.args.map_or_pdb_selection.value == 'PDB':
            self.launcher.add_file(
                arg_name=None,
                path=self.get_synthetic_map_name(),
                file_type='map',
                description='Synthetic map from PDB',
                selected=True)
        self.launcher.add_file(
            arg_name=None,
            path=diff_map1,
            file_type='map',
            description='Difference map 1 (map1 - map2)',
            selected=True)
        self.launcher.add_file(
            arg_name=None,
            path=diff_map2,
            file_type='map',
            description='Difference map 2 (map2 - map1)',
            selected=True)
        
        if self.args.save_fracmap.value:
            diff_fracmap1 = os.path.join(self.task.job_location,
                                 'diff_frac1.mrc')
            if os.path.exists(diff_fracmap1):
                self.launcher.add_file(
                    arg_name=None,
                    path=diff_fracmap1,
                    file_type='map',
                    description='Fractional difference map 1 (map1-map2)/map1',
                    selected=False)
            diff_fracmap2 = os.path.join(self.task.job_location,
                                 'diff_frac2.mrc')
            
            if os.path.exists(diff_fracmap2):
                self.launcher.add_file(
                    arg_name=None,
                    path=diff_fracmap2,
                    file_type='map',
                    description='Fractional difference map 2 (map2-map1)/map2',
                    selected=False)
        # Add power spectra plot
        power_spectra_plot = os.path.join(
            self.task.job_location,
            'spectra.png')
        if os.path.exists(power_spectra_plot):
            self.launcher.add_file(
                arg_name=None,
                path=power_spectra_plot,
                description='Matched power spectra of input maps',
                selected=False)

        message = ('Chimera will launch maps and automatically set input maps '
                   'to 2.0 sigma and difference maps to 1.0 sigma')
        self.launcher.set_message_label(message=message)

    def get_difference_map_names(self):
        m1_name =  os.path.basename(self.args.map_path_1.value).split('.')[0]
        if self.args.map_or_pdb_selection.value == 'Model':
            m2_name = self.args.pdb_path.value
        else:
            m2_name = self.args.map_path_2.value
        m2_name =  os.path.basename(m2_name).split('.')[0]
        diff_map1 = os.path.join(self.task.job_location,
                                 (m1_name + '-' + m2_name+'_diff.mrc'))
        diff_map2 = os.path.join(self.task.job_location,
                                 (m2_name + '-' + m1_name+'_diff.mrc'))
        return diff_map1, diff_map2

    def get_synthetic_map_name(self):
        syn_map_path = os.path.basename(self.args.pdb_path.value).split('.')[0]
        syn_map_path = os.path.join(self.task.job_location,
                                    syn_map_path + '_syn.mrc')
        return syn_map_path

    def run_chimera_custom(self):
        chimera_script = difference_map_chimera.__file__
        chimera_script = chimera_script.replace('.pyc', '.py')
        # Chimera script expects 2 input maps, followed by 2 difference maps
        if self.args.map_or_pdb_selection.value == 'Model':
            map2 = self.get_synthetic_map_name()
        else:
            map2 = self.args.map_path_2.value
        diff_map1, diff_map2 = self.get_difference_map_names()
        maps = [self.args.map_path_1.value,
                map2,
                diff_map1,
                diff_map2]
        map_args = chimera_script
        # Check map exists else pass 'None'
        for map_ in maps:
            map_path = 'None'
            if map_ is not None:
                if os.path.exists(map_):
                    map_path = map_
            map_args += ' ' + map_path
        if self.args.map_or_pdb_selection.value == 'PDB':
            map_args += (' ' + self.args.pdb_path.value)
        else:
            map_args += (' no_pdb')
        run_args = ['--script']
        run_args.append(map_args)
        process = QtCore.QProcess()
        chimera_bin = which('chimera')
        if chimera_bin is None:
            print 'Chimera executable not found (add executable to system PATH)'
        else:
            process.startDetached(chimera_bin, run_args)


def main():
    '''
    Launch standalone task runner.
    '''
    command_line_launch.ccpem_task_launch(
        task_class=difference_map_task.DifferenceMap,
        window_class=DifferenceMapWindow)

if __name__ == '__main__':
    main()
