
import os, sys, re
import argparse
import subprocess as SP

import pyrvapi_ext as API
from pyrvapi_ext import parsers

def main():
    parser = argparse.ArgumentParser(
        description='Test run of log parsers.'
    )
    parser.add_argument(
        'stdin',
        help='log file to parse',
        type=open,
        metavar='<file>'
    )
    parser.add_argument(
        '-d', '--doc-file',
        help='storred and/or restorred document path (relative or absolute)',
        default='storred_doc',
        type=str,
        metavar='<file>'
    )
    parser.add_argument(
        '-r', '--report-dir',
        help='report dir path (relative or absolute)',
        default='report',
        type=str,
        metavar='<file>'
    )
    parser.add_argument(
        '-k', '--kwdin',
        help='kwd file to parse',
        type=open,
        metavar='<file>'
    )
    parser.add_argument(
        '-c', '--custom',
        help='use customised parser',
        default='generic',
        type=str,
        metavar='<parser_prefix>'
    )
    parser.add_argument(
        '-s', '--split',
        help='separate folder for each individual program run',
        action='store_true'
    )
    parser.add_argument(
        '-e', '--echo',
        help='copy standard input to standard output',
        action='store_true'
    )
    parser.add_argument(
        '-v', '--verbose',
        help='show results of parsing for each recognised line',
        action='store_true'
    )
    parser.add_argument(
        '-p', '--flush-pause',
        help='''
          pause in seconds before each call pyrvapi_ext.flush
        ''',
        default=0,
        type=float,
        metavar='<float>'
    )
    parser.add_argument(
        '-a', '--action-pause',
        help='''
          pause in seconds after each regexp match
          and before any associaded action
        ''',
        default=0,
        type=float,
        metavar='<float>'
    )
    parser.add_argument(
        '-t', '--hide-tables',
        help='hide graph tables',
        action='store_true'
    )
    parser.add_argument(
        '-n', '--no-references',
        help='skip references',
        action='store_true'
    )
    opt = parser.parse_args()
    job_params = None
    if opt.kwdin:
        job_params = dict()
        for line in opt.kwdin:
            key, sep, value = line.partition('=')
            if sep == '=':
                job_params[key.strip()] = value.strip()

    if os.path.isfile(opt.doc_file):
      d = API.document.fromfile(
        docfile = opt.doc_file,
      )

    else:
      d = API.document.newdoc(
        wintitle = 'PyRVAPI Parsers Demo',
        reportdir = opt.report_dir,
        docfile = opt.doc_file,
      )
      d.add_header('Log Parsers Demo')

    API.flush()
    if opt.stdin.name == '/dev/null':
      d.close(False)

    else:
      log_parser = getattr(parsers, opt.custom + '_parser')
      tab = API.pyrvapi_tab(opt.stdin.name)
#
#     istream = opt.stdin
#     proc = SP.Popen(('cat', '-'), stdin=opt.stdin, stdout=SP.PIPE, stderr=SP.STDOUT)
      proc = SP.Popen(('cat', '-'), stdin=opt.stdin, stdout=SP.PIPE)
      istream = proc.stdout
#
      ostream = sys.stdout if opt.echo else None
      summary = dict()
      log_parser(
        tab.id,
        show_progs = opt.split,
        hide_refs = opt.no_references,
        job_params = job_params,
        pause = opt.flush_pause,
        summary = summary,
        graph_tables = not opt.hide_tables,
      ).parse_stream(
        istream,
        ostream = ostream,
        verbose = opt.verbose,
        pause = opt.action_pause,
        patches = [fix_fsc_table, fix_stars],
      )
      API.flush()
      sys.stdout.write('demo.py, summary:\n')
      for key, value in sorted(summary.items()):
        sys.stdout.write(key + ':\n')
        for kv in sorted(value.items()):
          sys.stdout.write('%s = %s\n' %kv)

      d.close()

def fix_fsc_table(line):
    return re.sub('sigmaSig ZZ TT cor(\|F1\|,\|F2\|) \$\$', 'sigmaSig ZZ TT cor(\|F1\|,\|F2\|) H1 H2 \$\$', line)
    '''
    error_line = ' 2sin(th)/l 2sin(th)/l NREF sigma  FSC PHdiff cos(PHdiff) sigmaSig ZZ TT cor(|F1|,|F2|) $$'
    fixed_line = ' 2sin(th)/l 2sin(th)/l NREF sigma  FSC PHdiff cos(PHdiff) sigmaSig ZZ TT cor(|F1|,|F2|) H1 H2 $$'
    '''

def fix_stars(line):
    return re.sub('0.0\*\*\*\*\*\*  0.00', '0.0  0.00  0.00', line)

if __name__ == '__main__':
    main()

